<?php declare(strict_types=1);

namespace Addons\Integrations\Pusher\Apps;

use App\Modules\Core\Models\Integration;
use BeyondCode\LaravelWebSockets\Apps\App;
use BeyondCode\LaravelWebSockets\Exceptions\InvalidApp;
use Illuminate\Contracts\Config\Repository;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

use function collect;

class AppProvider implements \BeyondCode\LaravelWebSockets\Apps\AppProvider
{
    /** @var Repository */
    private $config;

    /** @var Collection */
    private $settings;

    public function __construct(Repository $config)
    {
        $this->config = $config;
        $this->settings = $this->getSettings();
    }

    /**
     * @return mixed[]
     */
    public function all(): array
    {
        return $this->apps()
            ->map(function (array $appAttributes) {
                return $this->instanciate($appAttributes);
            })
            ->toArray();
    }

    /**
     * @param mixed $appId
     */
    public function findById($appId): ?App
    {
        $app = $this->apps()
            ->filter(function (array $item) use ($appId) {
                return $item['id'] === $appId;
            })
            ->first();

        return $this->instanciate($app);
    }

    public function findByKey(string $appKey): ?App
    {
        $app = $this->apps()
            ->filter(function (array $item) use ($appKey) {
                return $item['key'] === $appKey;
            })
            ->first();

        return $this->instanciate($app);
    }

    public function findBySecret(string $appSecret): ?App
    {
        $app = $this->apps()
            ->filter(function (array $item) use ($appSecret) {
                return $item['secret'] === $appSecret;
            })
            ->first();

        return $this->instanciate($app);
    }

    private function getSettings(): Collection
    {
        /** @var Collection $integrations */
        $integrations = $this->config->get('integration', collect());

        /** @var Integration|null $pusher */
        $pusher = $integrations
            ->filter(function ($item) {
                return Str::lower($item->name) === 'pusher';
            })
            ->first();

        return collect($pusher === null ? [] : $pusher->settings);
    }

    private function apps(): Collection
    {
        return collect([
            [
                'id' => $this->settings->get('app_id'),
                'name' => $this->config->get('app.name'),
                'key' => $this->settings->get('key'),
                'secret' => $this->settings->get('secret'),
                'path' => null,
                'capacity' => null,
                'enable_client_messages' => false,
                'enable_statistics' => false,
            ]
        ]);
    }

    /**
     * @param mixed[] $appAttributes
     * @return App|null
     * @throws InvalidApp
     */
    private function instanciate(?array $appAttributes): ?App
    {
        if (! $appAttributes) {
            return null;
        }

        $app = new App(
            $appAttributes['id'],
            $appAttributes['key'],
            $appAttributes['secret']
        );

        if (isset($appAttributes['name'])) {
            $app->setName($appAttributes['name']);
        }

        if (isset($appAttributes['host'])) {
            $app->setHost($appAttributes['host']);
        }

        if (isset($appAttributes['path'])) {
            $app->setPath($appAttributes['path']);
        }

        $app->enableClientMessages($appAttributes['enable_client_messages'])
            ->enableStatistics($appAttributes['enable_statistics'])
            ->setCapacity($appAttributes['capacity'] ?? null);

        return $app;
    }
}
