<?php declare(strict_types=1);

namespace Addons\Integrations\Pusher\Controllers;

use Addons\Integrations\Pusher\Listeners\AddMetaConfig;
use Addons\Integrations\Pusher\Listeners\AddPusherJs;
use Addons\Integrations\Pusher\Listeners\OverwriteEchoConfig;
use Addons\Integrations\Pusher\Requests\SettingsRequest;
use App\Broadcasting\PusherBroadcaster;
use App\Modules\Core\Controllers\Integrations\Integration;
use App\Modules\Core\Models\Integration as IntegrationModel;
use App\Modules\Core\Models\IntegrationService;
use Illuminate\Broadcasting\Broadcasters\PusherBroadcaster as IlluminatePusherBroadcaster;
use Illuminate\Broadcasting\BroadcastException;
use Illuminate\Broadcasting\BroadcastManager;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Log\Logger;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\View as ViewFacade;
use Illuminate\Support\Str;
use JsValidator;
use Psr\Log\LoggerInterface;
use Pusher\Pusher as PusherSdk;
use Pusher\PusherException;
use SupportPal\ProxyService\ProxyService;
use TemplateView;

use function app;
use function array_replace_recursive;
use function collect;
use function config;
use function is_numeric;
use function redirect;
use function response;
use function session;
use function sprintf;
use function trans;

class Pusher extends Integration
{
    /**
     * Integration identifier.
     *
     * @var string
     */
    const IDENTIFIER = 'Pusher';

    /**
     * Integration settings route.
     *
     * @var string
     */
    const SETTINGS_ROUTE = 'integration.pusher.settings';

    /**
     * Initialise the integration.
     */
    public function __construct()
    {
        parent::__construct();

        $this->setIdentifier(self::IDENTIFIER);

        // Register the settings page.
        $this->registerSetting(self::SETTINGS_ROUTE);

        $this->replacePusherDriver();
        $this->registerViewListeners();
    }

    /**
     * Get the settings page.
     *
     * @return View
     */
    public function getSettingsPage()
    {
        return TemplateView::other($this->getNamespace() . '::settings')
            ->with('docsLink', 'https://docs.supportpal.com/current/Third+Party+Integrations+Pusher')
            ->with('jsValidator', JsValidator::formRequest(SettingsRequest::class))
            ->with('settings', $this->getSettings());
    }

    /**
     * Update the settings.
     *
     * @param  SettingsRequest $request
     * @return RedirectResponse
     */
    public function updateSettings(SettingsRequest $request)
    {
        $data = $this->defaultParams($request->all(['app_id', 'key', 'secret', 'options']));

        // Work through each row of data.
        foreach (Arr::dot($data) as $key => $value) {
            if (! empty($value) || is_numeric($value)) {
                $this->addSetting((string) $key, $value);
            } else {
                $this->removeSetting((string) $key);
            }
        }

        // All done, return with a success message.
        session()->flash('success', trans('messages.success_settings'));

        return redirect()->route(self::SETTINGS_ROUTE);
    }

    /**
     * Validate the settings.
     *
     * @param  SettingsRequest $request
     * @return JsonResponse
     */
    public function validateSettings(SettingsRequest $request): JsonResponse
    {
        $data = $this->defaultParams($request->all(['app_id', 'key', 'secret', 'options']));

        try {
            app()->make(IlluminatePusherBroadcaster::class, ['pusher' => $this->createPusherSdk($data, app())])
                ->broadcast(['test-channel'], 'test-event');

            return response()->json([
                'status'  => 'success',
                'message' => null,
                'data'    => null,
            ]);
        } catch (BroadcastException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage(),
                'data'    => null,
            ]);
        }
    }

    /**
     * Plugins can run an installation routine when they are activated. This will typically include adding default
     * values, initialising database tables and so on.
     *
     * @return boolean
     */
    public function activate()
    {
        // Register the services that the integration can be used for.
        $this->registerService(IntegrationService::SERVICE_BROADCASTING);

        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still remain. This function should
     * typically clear any caches and temporary directories.
     *
     * @return boolean
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never was there. This function should
     * delete any created database tables, and any files created outside of the plugin directory.
     *
     * @return boolean
     */
    public function uninstall()
    {
        $this->removeSettings();

        return true;
    }

    private function replacePusherDriver(): void
    {
        /** @var IntegrationModel|null $integration */
        $integration = Config::get('integration')->where('id', $this->getModel('id'))->first();
        if ($integration === null) {
            return;
        }

        /** @var array<string,string> $settings */
        $settings = $integration->settings;

        // Not configured
        if (collect($settings)->isEmpty() || empty($settings['key'])) {
            return;
        }

        $driver = Str::lower($integration->name);

        // Update configuration.
        foreach ($settings as $setting => $value) {
            config()->set(sprintf('broadcasting.connections.%s.%s', $driver, $setting), $value);
        }

        config()->set('broadcasting.default', $driver);

        $broadcastManager = app()->make(BroadcastManager::class);
        $broadcastManager->extend($driver, function (Application $app, array $config) {
            return new PusherBroadcaster($this->createPusherSdk($config, $app), $app->make(Logger::class));
        });
    }

    private function registerViewListeners(): void
    {
        $key = config('broadcasting.connections.pusher.key');
        if ($key === null) {
            return;
        }

        ViewFacade::hook('operator.head', AddMetaConfig::class);
        ViewFacade::hook('operator.footer', AddPusherJs::class);

        ViewFacade::composer('operator.*.footer_common', OverwriteEchoConfig::class);
    }

    /**
     * @param mixed[] $data
     * @return mixed[]
     */
    private function defaultParams(array $data): array
    {
        // Start with config options.
        $data = array_replace_recursive(config('broadcasting.connections.pusher'), $data);

        $settings = $this->getSettings();

        // Set defaults if self-hosted.
        if ($data['options']['hosting'] !== '0') {
            if (empty($data['app_id'])) {
                $data['app_id'] = Arr::get($settings, 'app_id');
            }

            if (empty($data['key'])) {
                $data['key'] = Arr::get($settings, 'key');
            }

            if (empty($data['secret'])) {
                $data['secret'] = Arr::get($settings, 'secret');
            }

            unset($data['options']['cluster']);
        } else {
            // Clear values not needed for cloud.
            foreach (['host', 'external_host', 'port', 'encryption'] as $key) {
                unset($data['options'][$key]);
            }
        }

        return $data;
    }

    /**
     * @param mixed[] $config
     * @param Application $app
     * @return PusherSdk
     * @throws PusherException
     */
    private function createPusherSdk(array $config, $app): PusherSdk
    {
        $config['options']['curl_options'] = app()->make(ProxyService::class)->curlOpts();
        $pusher = new PusherSdk($config['key'], $config['secret'], $config['app_id'], $config['options']);
        $pusher->setLogger($app->make(LoggerInterface::class));

        return $pusher;
    }
}
