<?php
/**
 * File Steam.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace Addons\Integrations\Steam\Controllers;

use Addons\Integrations\Steam\Requests\SettingsRequest;
use App\Modules\Core\Controllers\Integrations\Integration;
use App\Modules\Core\Models\IntegrationService;
use Illuminate\Support\Arr;
use JsValidator;
use Lang;
use Redirect;
use Session;
use TemplateView;

/**
 * Class Steam
 */
class Steam extends Integration
{
    /**
     * Integration identifier.
     *
     * @var string
     */
    const IDENTIFIER = 'Steam';

    /**
     * Integration settings route.
     *
     * @var string
     */
    const SETTINGS_ROUTE = 'integration.steam.settings';

    /**
     * Initialise the integration.
     */
    public function __construct()
    {
        parent::__construct();

        $this->setIdentifier(self::IDENTIFIER);

        // Register the settings page.
        $this->registerSetting(self::SETTINGS_ROUTE);
    }

    /**
     * Get the settings page.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function getSettingsPage()
    {
        return TemplateView::other($this->getNamespace() . '::settings')
            ->with('docsLink', 'https://docs.supportpal.com/current/Third+Party+Integrations+Steam')
            ->with('jsValidator', JsValidator::formRequest(SettingsRequest::class))
            ->with('settings', ['brand-settings' => $this->getBrandSettings()])
            ->with('brands', brand_config(null));
    }

    /**
     * Update the settings.
     *
     * @param  SettingsRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateSettings(SettingsRequest $request)
    {
        $brands = brand_config(null);

        $inputs = [];

        // Add inputs for each brand.
        foreach ($brands as $brand) {
            array_push($inputs, 'brand-settings.' . $brand->id . '.client_secret');
        }

        $data = $request->all($inputs);

        // Work through each row of data.
        foreach (Arr::get($data, 'brand-settings', []) as $brandId => $values) {
            foreach ($values as $key => $value) {
                if (! empty($value) || is_numeric($value)) {
                    $this->addBrandSetting($brandId, $key, $value);
                } else {
                    $this->removeBrandSetting($brandId, $key);
                }
            }
        }

        // All done, return with a success message.
        Session::flash('success', Lang::get('messages.success_settings'));
        return Redirect::route(self::SETTINGS_ROUTE);
    }

    /**
     * Plugins can run an installation routine when they are activated. This will typically include adding default
     * values, initialising database tables and so on.
     *
     * @return boolean
     */
    public function activate()
    {
        // Register the services that the integration can be used for.
        $this->registerService(IntegrationService::SERVICE_SOCIAL_LOGIN);

        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still remain. This function should
     * typically clear any caches and temporary directories.
     *
     * @return boolean
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never was there. This function should
     * delete any created database tables, and any files created outside of the plugin directory.
     *
     * @return boolean
     */
    public function uninstall()
    {
        $this->removeSettings();

        return true;
    }
}
