<?php declare(strict_types=1);

namespace Addons\Integrations\Twilio\Controllers;

use Addons\Integrations\Twilio\Provider\Twilio as TwilioProvider;
use Addons\Integrations\Twilio\Requests\SettingsRequest;
use App\Modules\Core\Controllers\Integrations\Integration;
use App\Modules\Core\Controllers\PhoneVerify\Manager\Factory;
use App\Modules\Core\Models\IntegrationService;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Arr;
use JsValidator;
use Redirect;
use TemplateView;

use function app;
use function array_push;
use function is_numeric;
use function trans;

class Twilio extends Integration
{
    const IDENTIFIER = 'Twilio';
    const SERVICE_NAME = 'Twilio';

    const DOCS_LINK = 'https://docs.supportpal.com/current/Third+Party+Integrations+Twilio';

    const SETTINGS_ROUTE = 'integrations.twilio.settings';

    public function __construct()
    {
        parent::__construct();

        $this->setIdentifier(self::IDENTIFIER);

        if ($this->isVerifySidConfigured(current_brand())) {
            $this->registerTwilioVerifyProvider();
        }

        $this->registerSetting(self::SETTINGS_ROUTE);
    }

    public function activate()
    {
        $this->registerService(self::SERVICE_NAME);

        $this->registerService(IntegrationService::SERVICE_PHONE_VERIFY);

        return true;
    }

    public function deactivate()
    {
        return true;
    }

    public function uninstall()
    {
        return true;
    }

    public function getSettingsPage(): View
    {
        return TemplateView::other($this->getNamespace() . '::settings')
            ->with('jsValidator', JsValidator::formRequest(SettingsRequest::class))
            ->with('docsLink', self::DOCS_LINK)
            ->with('settings', ['brand-settings' => $this->getBrandSettings()])
            ->with('brands', brand_config(null));
    }

    public function updateSettings(SettingsRequest $request): RedirectResponse
    {
        $brands = brand_config(null);

        $inputs = [];

        // Add inputs for each brand.
        foreach ($brands as $brand) {
            array_push(
                $inputs,
                'brand-settings.' . $brand->id . '.account_sid',
                'brand-settings.' . $brand->id . '.auth_token',
                'brand-settings.' . $brand->id . '.api_key',
                'brand-settings.' . $brand->id . '.api_secret',
                'brand-settings.' . $brand->id . '.verify_sid',
            );
        }

        $data = $request->all($inputs);

        // Work through each row of data.
        foreach (Arr::get($data, 'brand-settings', []) as $brandId => $values) {
            foreach ($values as $key => $value) {
                if (! empty($value) || is_numeric($value)) {
                    $this->addBrandSetting($brandId, $key, $value);
                } else {
                    $this->removeBrandSetting($brandId, $key);
                }
            }
        }

        return Redirect::route(self::SETTINGS_ROUTE)
            ->with('success', trans('messages.success_settings'));
    }

    private function registerTwilioVerifyProvider(): void
    {
        app()->make(Factory::class)
            ->extend(self::IDENTIFIER, function ($app) {
                return $app->make(TwilioProvider::class);
            })
            ->setDefaultDriver(self::IDENTIFIER);
    }

    private function isVerifySidConfigured(int $brandId): bool
    {
        $settings = $this->getBrandSettings();
        if (! isset($settings[$brandId])) {
            return false;
        }

        return ! empty($settings[$brandId]['verify_sid']);
    }
}
