<?php declare(strict_types=1);

namespace Addons\Integrations\Twilio\Provider;

use Addons\Integrations\Twilio\Controllers\Twilio as TwilioIntegration;
use Addons\Integrations\Twilio\Factory\ClientFactory;
use App\Exceptions\Generic\RuntimeException;
use App\Modules\Core\Controllers\Integrations\Facades\IntegrationFactory;
use App\Modules\Core\Controllers\PhoneVerify\Exception\SendVerificationTokenFailedException;
use App\Modules\Core\Controllers\PhoneVerify\Provider\Provider;
use InvalidArgumentException;
use Log;
use Twilio\Exceptions\ConfigurationException;
use Twilio\Exceptions\TwilioException;
use Twilio\Rest\Client;

use function sprintf;

class Twilio extends Provider
{
    private TwilioIntegration $integration;

    private ClientFactory $clientFactory;

    private Client $twilioClient;

    private string $verifySid;

    /** @var mixed[] */
    private array $settings;

    /**
     * @throws ConfigurationException
     */
    public function __construct(ClientFactory $clientFactory)
    {
        $this->integration = $this->getIntegration();
        $this->clientFactory = $clientFactory;
        $this->settings = $this->integration->getBrandSettings();
        $this->setConfiguration();
    }

    public function sendVerificationCode(string $phoneNumber, string $mode = self::MODE_SMS): bool
    {
        if (! $this->supported($mode)) {
            throw new InvalidArgumentException('verify mode not supported.');
        }

        try {
            $verificationInstance = $this->twilioClient->verify
                ->v2
                ->services($this->verifySid)
                ->verifications
                ->create($phoneNumber, $mode);

            return $verificationInstance->status === 'pending';
        } catch (TwilioException $configurationException) {
            Log::error($configurationException->getMessage(), [$configurationException]);

            throw new SendVerificationTokenFailedException(
                $configurationException->getMessage(),
                0,
                $configurationException
            );
        }
    }

    public function verify(string $phoneNumber, string $code): bool
    {
        try {
            $verificationStatus = $this->twilioClient
                ->verify
                ->v2
                ->services($this->verifySid)
                ->verificationChecks
                ->create(['code' => $code, 'to' => $phoneNumber]);

            return $verificationStatus->status === 'approved';
        } catch (TwilioException $configurationException) {
            Log::error($configurationException->getMessage(), [$configurationException]);

            throw new SendVerificationTokenFailedException(
                $configurationException->getMessage(),
                0,
                $configurationException
            );
        }
    }

    private function getIntegration(): TwilioIntegration
    {
        $name = TwilioIntegration::IDENTIFIER;

        /** @var TwilioIntegration|null $integration */
        $integration = IntegrationFactory::getInstance($name);
        if ($integration === null || ! $integration->isEnabled()) {
            throw new RuntimeException(sprintf('Unable to find integration of name %s', $name));
        }

        return $integration;
    }

    /**
     * @throws ConfigurationException
     */
    private function setConfiguration(): void
    {
        $brandSettings = $this->settings[current_brand()] ?? null;
        foreach (['account_sid', 'auth_token', 'verify_sid'] as $key) {
            if (empty($brandSettings[$key])) {
                throw new RuntimeException(sprintf('An account SID, auth token and verification SID is required to create a client. %s is not set.', $key));
            }
        }

        $this->twilioClient = $this->clientFactory->create($brandSettings['account_sid'], $brandSettings['auth_token']);
        $this->verifySid = $brandSettings['verify_sid'];
    }
}
