<?php declare(strict_types=1);

/**
 * ActivityLog Widget
 *
 * @package    Addons\Widgets\ActivityLog\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.0.0
 */
namespace Addons\Widgets\ActivityLog\Controllers;

use App\Modules\Core\Controllers\Plugins\Plugin;
use App\Modules\Core\Controllers\Plugins\Widget;
use App\Modules\Core\Models\ActivityLog as ActivityLogModel;
use Datatable;
use Exception;
use Illuminate\Contracts\View\View as ViewContract;
use Illuminate\Http\JsonResponse;
use Lang;
use View;

use function now;
use function route;

/**
 * Class ActivityLog
 *
 * @package    Addons\Widgets\ActivityLog\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.0.0
 */
class ActivityLog extends Plugin implements Widget
{
    public function getEmbeddableView(): ?ViewContract
    {
        $table = Datatable::table()
            ->addColumn(
                Lang::choice('general.event', 1),
                Lang::get('general.name'),
                Lang::get('general.when')
            )
            ->setUrl(route('widgets.activitylog.table'))
            ->setOptions([
                'sDom'        => 'rt',
                'order'       => [[3, 'desc']],
                'pageLength'  => 5,
                'columnDefs'  => [
                    ['targets' => 0, 'className' => 'all', 'orderable' => false],
                    ['targets' => 1, 'className' => 'desktop', 'orderable' => false],
                    ['targets' => 2, 'className' => 'min-tablet-p', 'orderable' => false],
                    ['targets' => 3, 'visible' => false]
                ]
            ])
            ->noScript();

        return View::make('Widgets#ActivityLog::view', ['table' => $table]);
    }

    /**
     * Datatable Json Response.
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function dataTable()
    {
        $collection = ActivityLogModel::with('user')
            ->whereAssignedBrands()
            ->where('created_at', '>=', now()->subMonth()->getTimestamp())
            ->where('created_at', '<=', now()->getTimestamp());

        return Datatable::query($collection)
            ->addColumn('event', function (ActivityLogModel $model) {
                return $model->formatEvent();
            })
            ->addColumn('name', function (ActivityLogModel $model) {
                return <<<EOT
{$model->formatName()}
<span class="sp-description">({$model->typeToString()})</span>
EOT;
            })
            ->addColumn('when', function (ActivityLogModel $model) {
                return timeago($model->created_at);
            })
            ->addColumn('created_at', function (ActivityLogModel $model) {
                return $model->created_at;
            })
            ->make();
    }

    /**
     * Plugins can run an installation routine when they are activated. This
     * will typically include adding default values, initialising database tables
     * and so on.
     *
     * @return mixed
     */
    public function activate()
    {
        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still
     * remain. This function should typically clear any caches and temporary directories.
     *
     * @return mixed
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never
     * was there. This function should delete any created database tables, and any files
     * created outside of the plugin directory.
     *
     * @return mixed
     */
    public function uninstall()
    {
        return true;
    }
}
