<?php
/**
 * System Overview Widget
 *
 * @package    Addons\Widgets\SystemOverview\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.0.0
 */
namespace Addons\Widgets\SystemOverview\Controllers;

use App\Modules\Core\Controllers\Plugins\Plugin;
use App\Modules\Core\Controllers\Plugins\Widget;
use App\Modules\Ticket\Models\Ticket;
use Carbon\Carbon;
use Config;
use DB;
use Illuminate\Contracts\View\View as ViewContract;
use Lang;
use View;

/**
 * Class SystemOverview
 *
 * @package    Addons\Widgets\SystemOverview\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.0.0
 */
class SystemOverview extends Plugin implements Widget
{
    public function getEmbeddableView(): ?ViewContract
    {
        $records = [
            [
                'name'  => Lang::get('ticket.overdue'),
                'slug'  => 'overdue',
                'count' => $this->countOverdueTickets(),
                'link'  => route('ticket.operator.ticket', [
                    'duebefore' => time()
                ]),
            ], [
                'name'  => Lang::get('general.open'),
                'slug'  => 'open',
                'count' => $this->countOpenTickets(),
                'link'  => route('ticket.operator.ticket', [
                    'status' => Config::get('settings.default_open_status')
                ])
            ], [
                'name'  => Lang::get('general.pending'),
                'slug'  => 'pending',
                'count' => $this->countTicketsPending(),
                'link'  => route('ticket.operator.ticket', [
                    'status'     => Config::get('settings.default_open_status')
                        . (Config::get('settings.default_open_status') != Config::get('settings.default_reply_status')
                            ? ',' . Config::get('settings.default_reply_status') : ''),
                    'assignedto' => auth_user()->id
                ]),
            ], [
                'name'  => Lang::get('ticket.due_today'),
                'slug'  => 'due-today',
                'count' => $this->countTicketsDueToday(),
                'link'  => route('ticket.operator.ticket', [
                    'duebefore' => Carbon::tomorrow(getTimezone())->timestamp,
                    'dueafter'  => Carbon::today(getTimezone())->timestamp
                ]),
            ], [
                'name'  => Lang::get('ticket.unassigned'),
                'slug'  => 'unassigned',
                'count' => $this->countUnassignedTickets(),
                'link'  => route('ticket.operator.ticket', [
                    'status'     => Config::get('settings.default_open_status')
                        . (Config::get('settings.default_open_status') != Config::get('settings.default_reply_status')
                            ? ',' . Config::get('settings.default_reply_status') : ''),
                    'assignedto' => -1
                ]),
            ]
        ];

        return View::make('Widgets#SystemOverview::view', ['records' => $records]);
    }

    /**
     * Plugins can run an installation routine when they are activated. This
     * will typically include adding default values, initialising database tables
     * and so on.
     *
     * @return mixed
     */
    public function activate()
    {
        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still
     * remain. This function should typically clear any caches and temporary directories.
     *
     * @return mixed
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never
     * was there. This function should delete any created database tables, and any files
     * created outside of the plugin directory.
     *
     * @return mixed
     */
    public function uninstall()
    {
        return true;
    }

    /**
     * Fetch the number of overdue tickets
     *
     * @param  \Illuminate\Database\Query\Builder $query
     * @return int
     */
    public function getCount($query)
    {
        return DB::table(DB::raw("({$query->toSql()} LIMIT 1000) as sub"))
            ->mergeBindings($query->getQuery())
            ->count();
    }

    /**
     * Fetch the number of overdue tickets
     *
     * @return int
     */
    public function countOverdueTickets()
    {
        $query = Ticket::select(DB::raw(1))
            ->whereAssignedDepartmentAndBrand()
            ->overDue();

        return $this->getCount($query);
    }

    /**
     * Get the number of open tickets
     *
     * @return int
     */
    public function countOpenTickets()
    {
        $query = Ticket::select(DB::raw(1))
            ->whereAssignedDepartmentAndBrand()
            ->whereStatus([Config::get('settings.default_open_status')]);

        return $this->getCount($query);
    }

    /**
     * Get the number of tickets due today
     *
     * @return int
     */
    public function countTicketsDueToday()
    {
        $query = Ticket::select(DB::raw(1))
            ->whereAssignedDepartmentAndBrand()
            ->dueToday();

        return $this->getCount($query);
    }

    /**
     * Get the number of assigned tickets
     *
     * @return int
     */
    public function countUnassignedTickets()
    {
        $query = Ticket::select(DB::raw(1))
            ->whereAssignedDepartmentAndBrand()
            ->leftJoin('ticket_operator_membership', 'ticket.id', '=', 'ticket_operator_membership.ticket_id')
            ->whereNull('ticket_operator_membership.ticket_id')
            ->whereStatus([
                Config::get('settings.default_open_status'),
                Config::get('settings.default_reply_status')
            ]);

        return $this->getCount($query);
    }

    /**
     * Get the number of tickets that are pending (assigned to operator and open)
     *
     * @return int
     */
    public function countTicketsPending()
    {
        $query = Ticket::select(DB::raw(1))
            ->whereAssignedDepartmentAndBrand()
            ->join('ticket_operator_membership', 'ticket.id', '=', 'ticket_operator_membership.ticket_id')
            ->where('ticket_operator_membership.user_id', auth_user()->id)
            ->whereStatus([
                Config::get('settings.default_open_status'),
                Config::get('settings.default_reply_status')
            ]);

        return $this->getCount($query);
    }
}
