<?php declare(strict_types=1);

namespace Addons\Widgets\Todo\Controllers;

use App\Exceptions\Database\SQLException;
use App\Modules\Core\Controllers\Plugins\Plugin;
use App\Modules\Core\Controllers\Plugins\Widget;
use Addons\Widgets\Todo\Models\Todo as TodoModel;
use Addons\Widgets\Todo\Requests\CreateOrUpdateRequest;
use Datatable;
use Exception;
use Illuminate\Contracts\View\View as ViewContract;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use View;

use function count;
use function e;
use function implode;
use function now;
use function response;
use function route;
use function sprintf;
use function version_compare;

class Todo extends Plugin implements Widget
{
    const IDENTIFIER = 'Todo';

    /**
     * List of inputs expected from the form
     *
     * @var string[]
     */
    private $inputs = ['id', 'text', 'due', 'complete'];

    public function getEmbeddableView(): ?ViewContract
    {
        $table = Datatable::table()
            ->addColumn(
                Lang::get('general.due'),
                Lang::get('general.description'),
                '',
                ''
            )
            ->setUrl(route('widgets.todo.table'))
            ->setOptions([
                'sDom'        => 'rtp',
                'columnDefs'  => [
                    ['targets' => [0, 2, 3], 'className' => 'all'],
                    ['targets' => 1, 'className' => 'min-tablet-p'],
                    ['targets' => [2, 3], 'width' => '18px', 'orderable' => false]
                ]
            ])
            ->noScript();

        return View::make('Widgets#Todo::view')
            ->with('table', $table);
    }

    /**
     * Return DataTable data.
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function dataTable()
    {
        $collection = TodoModel::where('user_id', auth_user()->id)->orderBy('due')->get();

        return Datatable::collection($collection)
            ->orderColumns('due', 'text')
            ->setOrderStrip(true)
            ->setRowData(function ($model) {
                return [
                    'id'      => $model->id,
                    'due'     => formatDate($model->due, true),
                    'overdue' => now()->getTimestamp() > $model->due,
                ];
            })
            ->setRowClass(function ($model) {
                $classes = [];
                if (now()->getTimestamp() > $model->due) {
                    $classes[] = 'sp-text-red-600';
                }

                return implode(' ', $classes);
            })
            ->addColumn('due', function ($model) {
                return e(formatDate($model->due, true));
            })
            ->addColumn('text', function ($model) {
                return e($model->text);
            })
            ->addColumn('edit', function ($model) {
                return sprintf(
                    '<a data-route="%s" class="editTodo"><i class="fas fa-fw fa-edit"></i></a>',
                    e(route('widgets.todo.addTodo'))
                );
            })
            ->addColumn('mark', function ($model) {
                return sprintf(
                    '<a data-route="%s" class="markTodo"><i class="fas fa-fw fa-check"></i>',
                    e(route('widgets.todo.mark', ''))
                );
            })
            ->make();
    }

    /**
     * Plugins can run an installation routine when they are activated. This
     * will typically include adding default values, initialising database tables
     * and so on.
     *
     * @return mixed
     */
    public function activate()
    {
        // Create table if it doesn't exist
        if (! Schema::hasTable('operator_todo_widget')) {
            Schema::create('operator_todo_widget', function (Blueprint $table) {
                $table->engine = 'InnoDB';
                $table->charset = 'utf8mb4';
                $table->collation = 'utf8mb4_unicode_ci';

                $table->increments('id')->unsigned();
                $table->integer('user_id')->unsigned();
                $table->foreign('user_id')->references('id')->on('user')->onDelete('cascade');

                $table->text('text');
                $table->integer('due');
                $table->tinyInteger('complete')->default(0)->comment('0 = not complete, 1 = complete');

                $table->integer('created_at');
                $table->integer('updated_at')->default(0);
            });
        }

        // Added in version 1.1
        if ($this->installedVersion() === null || version_compare('1.1', $this->installedVersion()) === 1) {
            Schema::table('operator_todo_widget', function (Blueprint $table) {
                $table->dropColumn('complete');
                $table->integer('deleted_at')->nullable();
            });
        }

        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still
     * remain. This function should typically clear any caches and temporary directories.
     *
     * @return mixed
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never
     * was there. This function should delete any created database tables, and any files
     * created outside of the plugin directory.
     *
     * @return mixed
     */
    public function uninstall()
    {
        // Drop table if it exists
        if (Schema::hasTable('operator_todo_widget')) {
            Schema::disableForeignKeyConstraints();
            Schema::drop('operator_todo_widget');
            Schema::enableForeignKeyConstraints();
        }

        return true;
    }

    /**
     * Create or update a to-do item
     *
     * @param CreateOrUpdateRequest $request
     * @return JsonResponse
     */
    public function createOrUpdate(CreateOrUpdateRequest $request)
    {
        $data = $request->all($this->inputs);

        try {
            $todo = TodoModel::firstOrNew(['id' => $data['id']]);
            $todo->fill($data);
            $todo->user_id = isset($todo->user_id) ? $todo->user_id : auth_user()->id;

            // Save the record (if anything changed)
            if (count($todo->getDirty()) && ! $todo->save()) {
                throw new SQLException('Failed to write to database.');
            }

            return response()->json([
                'status'  => 'success',
                'data'    => $todo->toArray(),
                'message' => Lang::get('messages.success_updated', ['item' => Lang::get('general.record')])
            ], 200);
        } catch (SQLException $e) {
            Log::error($e);

            return response()->json([
                'status'  => 'error',
                'data'    => [],
                'message' => Lang::get('messages.error_updated', ['item' => Lang::get('general.record')])
            ], 200);
        }
    }

    /**
     * Mark an item as complete.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function mark(int $id)
    {
        try {
            /** @var TodoModel $todo */
            $todo = TodoModel::findOrFail($id);
            if (! $todo->delete()) {
                throw new SQLException('Failed to write to database.');
            }

            return response()->json([
                'status'  => 'success',
                'message' => Lang::get('messages.success_updated', ['item' => Lang::get('general.record')])
            ], 200);
        } catch (SQLException $e) {
            Log::error($e);

            return response()->json([
                'status'  => 'error',
                'message' => Lang::get('messages.error_updated', ['item' => Lang::get('general.record')])
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status'  => 'error',
                'message' => Lang::get('messages.error_notfound', ['item' => Lang::get('general.record')])
            ], 200);
        }
    }

    /**
     * Remove a to-do entry
     *
     * @param  int $id
     * @return mixed
     */
    public function delete(int $id)
    {
        try {
            /** @var TodoModel $record */
            $record = TodoModel::findOrFail($id);

            if (! $record->forceDelete()) {
                throw new SQLException('Failed to delete from database.');
            }

            return response()->json([
                'status'  => 'success',
                'data'    => $record->toArray(),
                'message' => Lang::get('messages.success_deleted', ['item' => Lang::get('general.record')])
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status'  => 'error',
                'data'    => [],
                'message' => Lang::get('messages.error_notfound', ['item' => Lang::get('general.record')])
            ], 404);
        } catch (SQLException $e) {
            Log::error($e);

            return response()->json([
                'status'  => 'error',
                'data'    => [],
                'message' => Lang::get('messages.error_deleted', ['item' => Lang::get('general.record')])
            ], 500);
        }
    }
}
