<?php declare(strict_types=1);

namespace Addons\Widgets\VersionCheck\Controllers;

use App\Modules\Core\Controllers\Plugins\Plugin;
use App\Modules\Core\Controllers\Plugins\Widget;
use App\Modules\Core\Controllers\Update\Manager\VersionManager;
use App\Modules\Core\Models\Repository\SettingsRepository;
use App\Modules\Core\Models\Setting;
use App\Modules\User\Models\OperatorSetting;
use App\Modules\User\Models\User;
use Cache;
use Config;
use Illuminate\Contracts\View\View as ViewContract;
use Illuminate\Http\JsonResponse;
use Response;
use SupportPal\Licensing\Http\Api\License\Information;
use SupportPal\Licensing\Models\License;
use SupportPal\Licensing\Utils\Version;
use View;

use function auth_user;
use function version_compare;

class VersionCheck extends Plugin implements Widget
{
    protected SettingsRepository $settingsRepository;

    protected Information $infoApi;

    protected Version $version;

    protected VersionManager $versionManager;

    public function __construct(
        SettingsRepository $settingsRepository,
        Information $infoApi,
        Version $version,
        VersionManager $versionManager
    ) {
        parent::__construct();

        $this->settingsRepository = $settingsRepository;
        $this->infoApi = $infoApi;
        $this->version = $version;
        $this->versionManager = $versionManager;
    }

    public function getEmbeddableView(): ?ViewContract
    {
        $versions = [
            'latest'    => $this->versionManager->getLatestVersion(),
            'installed' => $this->settingsRepository->getInstalledVersion(),
            'update'    => 0
        ];

        // Are we showing the new version popup - default value
        $popup = false;

        // Check if there's an update available
        if (isset($versions['latest']) && ! $this->versionManager->isRunningLatestVersion()) {
            // Update available and likely want to show popup
            $versions['update'] = 1;
            $popup = true;

            /** @var User $user */
            $user = auth_user();

            // Check if operator has acknowledged it already (and no newer version is available yet)
            if (($version = Config::get('settings.operator' . $user->id . '.widget_version_check'))
                && version_compare($version, $versions['latest']) >= 0
            ) {
                $popup = false;
            }
        } elseif ($this->version->isPreRelease()) {
            // Pre-release
            $versions['update'] = 2;
        }

        $licenseInformation = null;
        if (($key = License::key()) !== null) {
            $licenseInformation = $this->infoApi->get($key);
        }

        return View::make('Widgets#VersionCheck::view')
            ->with('versions', $versions)
            ->with('popup', $popup)
            ->with('licenseInformation', $licenseInformation);
    }

    /**
     * Set version check value for operator in database so they do not see popup again.
     */
    public function acknowledgePopup(): JsonResponse
    {
        /** @var User $user */
        $user = auth_user();

        // Update operator setting value
        OperatorSetting::where('user_id', $user->id)
            ->update(['widget_version_check' => $this->versionManager->getLatestVersion()]);

        // Clear the settings from the cache
        Cache::forget(Setting::CACHE_KEY);

        return Response::json(['status' => 'success']);
    }

    /**
     * Plugins can run an installation routine when they are activated. This
     * will typically include adding default values, initialising database tables
     * and so on.
     *
     * @return mixed
     */
    public function activate()
    {
        return true;
    }

    /**
     * Deactivating serves as temporarily disabling the plugin, but the files still
     * remain. This function should typically clear any caches and temporary directories.
     *
     * @return mixed
     */
    public function deactivate()
    {
        return true;
    }

    /**
     * When a plugin is uninstalled, it should be completely removed as if it never
     * was there. This function should delete any created database tables, and any files
     * created outside of the plugin directory.
     *
     * @return mixed
     */
    public function uninstall()
    {
        return true;
    }
}
