<?php

namespace SupportPal\Requirements;

use Closure;
use SupportPal\Requirements\Translation\Translation;

class Requirement
{
    /**
     * Human readable name for the requirement.
     *
     * @var Translation
     */
    private $name = '';

    /**
     * The message to show if the requirement is valid
     *
     * @var string
     */
    private $successMsg;

    /**
     * The message to show if the requirement is not valid
     *
     * @var string
     */
    private $failureMsg;

    /**
     * Function to check if the server meets the requirement.
     *
     * @var Closure
     */
    private $callback;

    /**
     * Whether the requirement is valid or not - this stops us validating over and over (within the same request).
     *
     * @var bool
     */
    private $valid;

    /**
     * Initialise a new system requirement.
     *
     * @param  Closure      $callback
     * @param  Translation  $name
     * @param  Translation  $successMsg (optional)
     * @param  Translation  $failureMsg (optional)
     */
    public function __construct(
        Closure $callback,
        Translation $name,
        Translation $failureMsg = null,
        Translation $successMsg = null
    ) {
        $this->name = $name;
        $this->callback = $callback;
        $this->successMsg = empty($successMsg) ? new Translation('messages.success') : $successMsg;
        $this->failureMsg = empty($failureMsg) ? new Translation('installer.not_available') : $failureMsg;
    }

    /**
     * Call the user provided call back that checks whether the server
     * meets system requirement.
     *
     * @param  bool $force
     * @return bool FALSE on error, if the callback result is not boolean
     */
    public function isValid($force = false)
    {
        if (! $force && $this->valid !== null) {
            return $this->valid;
        }

        // Check the requirement.
        $result = call_user_func($this->callback, $this);

        $this->valid = ! is_bool($result) ? false : $result;

        return $this->valid;
    }

    /**
     * Get the requirements name
     *
     * @return Translation
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Get the message to show if the requirement is valid
     *
     * @return string
     */
    public function getSuccessMsg()
    {
        return $this->successMsg;
    }

    /**
     * Get the message to show if the requirement is not valid
     *
     * @return Translation
     */
    public function getFailureMsg()
    {
        return $this->failureMsg;
    }

    /**
     * Set a new failure message.
     *
     * @param  string $message
     * @return void
     */
    public function setFailureMsg($message)
    {
        $this->failureMsg = $message;
    }
}
