<?php declare(strict_types=1);

namespace SupportPal\Core\Database\Repositories;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use InvalidArgumentException;
use SupportPal\Core\Database\Models\BaseModel;
use SupportPal\Core\Database\Repositories\Conditions\BaseCondition;

use function app;

class BaseRepository
{
    /** @var BaseModel */
    protected $model;

    /** @var Builder */
    protected $builder;

    /**
     * BaseRepository constructor.
     */
    public function __construct(string $model)
    {
        $this->model = app($model);

        if (! $this->model instanceof BaseModel) {
            throw new InvalidArgumentException(
                'Argument 1 must be an instance of \\SupportPal\\Core\\Models\\BaseModel'
            );
        }

        $this->builder = $this->model->newQuery();
    }

    /**
     * @param string[] $columns
     */
    public function get(array $columns = ['*']): Collection
    {
        return $this->builder->get($columns);
    }

    /**
     * @param string[] $columns
     */
    public function find(int $id, array $columns = ['*']): BaseModel
    {
        return $this->builder->find($id, $columns);
    }

    /**
     * @param string[] $columns
     * @throws ModelNotFoundException
     */
    public function findOrFail(int $id, array $columns = ['*']): BaseModel
    {
        return $this->builder->findOrFail($id, $columns);
    }

    /**
     * @param string[] $columns
     * @throws ModelNotFoundException
     */
    public function firstOrFail(array $columns = ['*']): BaseModel
    {
        return $this->builder->firstOrFail($columns);
    }

    public function paginate(int $perPage = 50): LengthAwarePaginator
    {
        return $this->builder->paginate($perPage);
    }

    /**
     * @param string[] $relations
     */
    public function with(array $relations): self
    {
        $this->builder->with($relations);

        return $this;
    }

    public function has(string $relation): self
    {
        $this->builder->has($relation);

        return $this;
    }

    public function applyCondition(BaseCondition $condition): BaseRepository
    {
        $condition->apply($this->builder);

        return $this;
    }
}
