<?php declare(strict_types=1);

/**
 * File Enum
 *
 * @package    SupportPal\Core\Http\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.0.0
 */
namespace SupportPal\Core\Http\Controllers;

use ReflectionClass;
use ReflectionException;

use function array_key_exists;
use function array_keys;
use function array_map;
use function array_values;
use function get_called_class;
use function in_array;
use function strtolower;

/**
 * Class Enum
 *
 * @package    SupportPal\Core\Http\Controllers
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.0.0
 */
abstract class Enum
{
    /**
     * Array with constants cached
     *
     * @var null
     */
    private static $constCacheArray = null;

    /**
     * Get a list of defined enums
     *
     * @return array
     *
     * @throws ReflectionException
     */
    private static function getConstants()
    {
        if (self::$constCacheArray === null) {
            self::$constCacheArray = [];
        }

        $calledClass = get_called_class();
        if (! array_key_exists($calledClass, self::$constCacheArray)) {
            $reflect = new ReflectionClass($calledClass);
            self::$constCacheArray[$calledClass] = $reflect->getConstants();
        }

        return self::$constCacheArray[$calledClass];
    }

    /**
     * Check if a enum exists for the specified name
     *
     * @param string $name
     * @param bool $strict
     * @return bool
     *
     * @throws ReflectionException
     */
    public static function isValidName($name, $strict = false)
    {
        $constants = self::getConstants();

        if ($strict) {
            return array_key_exists($name, $constants);
        }

        $keys = array_map('strtolower', array_keys($constants));

        return in_array(strtolower($name), $keys);
    }

    /**
     * Check if the value represents an enum
     *
     * @param string $value
     * @return bool
     *
     * @throws ReflectionException
     */
    public static function isValidValue($value)
    {
        $values = array_values(self::getConstants());

        return in_array($value, $values, $strict = true);
    }
}
