<?php

namespace Supportpal\HelpdeskBaseRequirements\Requirements;

use SupportPal\Requirements\Requirement;
use SupportPal\Requirements\RequirementGroup;
use SupportPal\Requirements\Translation\Translation;

class PhpExtensions extends AbstractRequirement
{
    public function getRequirementGroup()
    {
        $disabled = new Translation('installer.disabled');
        $enabled = new Translation('installer.enabled');

        $required = array(
            array('name' => 'BCMath'),
            array('name' => 'Ctype'),
            array('name' => 'cURL'),
            array('name' => 'DOM'),
            array('name' => 'FileInfo'),
            array('name' => 'GD'),
            array('name' => 'Hash'),
            array('name' => 'Iconv'),
            array('name' => 'JSON'),
            array('name' => 'Mbstring'),
            array('name' => 'OpenSSL'),
            array('name' => 'PDO'),
            array('name' => 'PDO_MySQL'),
            array('name' => 'Tokenizer'),
            array(
                'name' => 'XML',
                'callback' => function ($extension) {
                    return extension_loaded(strtolower($extension['name']))
                        && defined('LIBXML_DOTTED_VERSION')
                        && version_compare(LIBXML_DOTTED_VERSION, '2.7.0', '>=');
                }
            ),
        );

        $optional = array(
            array('name' => 'LDAP'),
            array('name' => 'IMAP'),
            array('name' => 'Gettext'),
            array('name' => 'Date'),
            array('name' => 'PCNTL'),
            array('name' => 'POSIX'),
            array('name' => 'Zip'),
            array('name' => 'Zlib'),
        );

        // Disable certain requirements in the CI environment.
        if (! $this->isTestEnv()) {
            $required[] = $this->ioncubeLoaders($disabled);
        }

        return new RequirementGroup(
            new Translation('installer.php_extensions'),
            $this->makeExtensionRequirements($required, $disabled, $enabled),
            $this->makeExtensionRequirements($optional, $disabled, $enabled),
            null,
            new Translation('installer.help_php_extensions')
        );
    }

    /**
     * Make an array of extension requirements.
     *
     * @param  array  $extensions
     * @param  Translation $failureMsg
     * @param  Translation $successMsg
     * @return array
     */
    private function makeExtensionRequirements(array $extensions, $failureMsg, $successMsg)
    {
        // Sort the array by name.
        $keys = array_map(function ($element) {
            return ucfirst($element['name']);
        }, $extensions);
        array_multisort($keys, $extensions);

        $result = array();
        foreach ($extensions as $extension) {
            $failureMessage = $failureMsg;
            if (isset($extension['error_msg'])) {
                $failureMessage = is_callable($extension['error_msg'])
                    ? call_user_func_array($extension['error_msg'], array($extension))
                    : $extension['error_msg'];
            }

            $successMessage = $successMsg;
            if (isset($extension['success_msg'])) {
                $successMessage = is_callable($extension['success_msg'])
                    ? call_user_func_array($extension['success_msg'], array($extension))
                    : $extension['success_msg'];
            }

            $result[] = new Requirement(
                function () use ($extension) {
                    $valid = true;
                    if (isset($extension['callback'])) {
                        $valid = call_user_func_array($extension['callback'], array($extension));
                    }

                    return extension_loaded(strtolower($extension['name'])) && $valid;
                },
                new Translation($extension['name']),
                $failureMessage,
                $successMessage
            );
        }

        return $result;
    }

    /**
     * Ioncube Loader extension requirement.
     *
     * @param  Translation $disabled
     * @return mixed[]
     */
    private function ioncubeLoaders(Translation $disabled)
    {
        $requiredIoncube = '110001';
        if (version_compare(phpversion(), '8.1', '>=')) {
            $requiredIoncube = '120000';
        }

        $expected = $this->getIonCubeVersion($requiredIoncube);
        $actual   = $this->getIonCubeVersion();

        return array(
            'name'      => 'ionCube Loader',
            'error_msg' => function ($extension) use ($disabled, $expected, $actual) {
                if (! extension_loaded(strtolower($extension['name']))) {
                    return $disabled;
                }

                return new Translation('installer.ioncube_version', array(
                    'required' => $expected,
                    'version'  => $actual
                ));
            },
            'callback'  => function ($extension) use ($requiredIoncube) {
                return ! function_exists('ioncube_loader_iversion')
                    || ioncube_loader_iversion() >= $requiredIoncube;
            }
        );
    }

    /**
     * Get or transform ionCube version to a user-friendly format.
     *
     * @param string|null $version
     * @return string
     */
    private static function getIonCubeVersion($version = null)
    {
        if ($version === null) {
            if (! function_exists('ioncube_loader_iversion')) {
                return '';
            }

            $version = ioncube_loader_iversion();
        }

        $majorVersion = floor($version / 10000);
        $minorVersion = floor((int) substr($version, -4) / 100);
        $patchVersion = (int) substr($version, -2);

        return sprintf('%d.%d.%d', $majorVersion, $minorVersion, $patchVersion);
    }
}