<?php

namespace Supportpal\HelpdeskBaseRequirements\Requirements;

use SupportPal\Requirements\Requirement;
use SupportPal\Requirements\RequirementGroup;
use SupportPal\Requirements\Translation\Translation;

class PhpSettings extends AbstractRequirement
{
    /**
     * List of php setting requirements.
     *
     * @return RequirementGroup
     */
    public function getRequirementGroup()
    {
        $required = $optional = array();
        $optional[] = new Requirement(function () {
            return defined('AF_INET6');
        }, new Translation('installer.ipv6_support'), new Translation('installer.ipv6_failure'));
        $optional[] = new Requirement(function () {
            if (! function_exists('ini_get')) {
                return false;
            }

            return (bool) ini_get('allow_url_fopen');
        }, new Translation('installer.allow_url_fopen'), new Translation('installer.allow_url_fopen_failure'));

        $required[] = new Requirement(function () {
            return extension_loaded('gd') && imagetypes() & IMG_PNG && imagetypes() & IMG_JPG;
        }, new Translation('installer.png_jpg_support'));

        $required[] = new Requirement(function () {
            if (! function_exists('ini_get')) {
                return false;
            }

            $disabled = array_map('trim', explode(',', ini_get('disable_functions')));

            return ! in_array('ini_set', $disabled);
        }, new Translation('installer.disabled_functions'));

        $required[] = new Requirement(function () {
            if (! function_exists('ini_get')) {
                return false;
            }

            // Handle both unlimited memory (-1) and check defined bytes are greater than our requirement of 128MB.
            $memory_limit = return_bytes(ini_get('memory_limit'));

            return $memory_limit == -1 || $memory_limit >= (128 * 1024 * 1024);
        }, new Translation('installer.memory_limit'), $this->getMemoryLimitFailureMsg());

        return new RequirementGroup(
            new Translation('installer.php_settings'),
            $required,
            $optional,
            function ($group) {
                if ($group->allValid()) {
                    return new Translation(
                        "installer.both_requirements",
                        array(
                            "required"       => $group->countValid($group->getRequired()),
                            "total_required" => count($group->getRequired()),
                            "optional"       => $group->countValid($group->getOptional()),
                            "total_optional" => count($group->getOptional()),
                        )
                    );
                }

                return new Translation('installer.help_php_settings');
            }
        );
    }

    /**
     * Get the memory_limit failure message.
     *
     * @return Translation
     */
    private function getMemoryLimitFailureMsg()
    {
        if (! function_exists('ini_get')) {
            $memory_limit = 'unknown';
        } else {
            $memory_limit = ini_get('memory_limit');
        }

        return new Translation('installer.memory_limit_error', array("limit" => $memory_limit));
    }
}
