<?php

namespace Supportpal\HelpdeskBaseRequirements\Requirements;

use Composer\Semver\Semver;
use Composer\Semver\VersionParser;
use SupportPal\Requirements\Requirement;
use SupportPal\Requirements\RequirementGroup;
use SupportPal\Requirements\Translation\Translation;
use UnexpectedValueException;

class PhpVersion extends AbstractRequirement
{
    /**
     * Make the php version requirement.
     *
     * @return RequirementGroup
     */
    public function getRequirementGroup()
    {
        // Must match Composer version constraint format
        // https://getcomposer.org/doc/articles/versions.md#writing-version-constraints
        $name = '>=7.4.0 <8.0 || >=8.1';
        if ($this->isTestEnv()) {
            // Strip off the maximum version constraint in test environment, allows us to test against
            // versions that we don't officially support yet.
            $name = '>=7.4.0';
        }

        $version = $this->getPhpVersion();

        return new RequirementGroup(
            new Translation('installer.php_version'),
            array(
                new Requirement(
                    function () use ($version, $name) {
                        return Semver::satisfies($version, $name);
                    },
                    new Translation($name),
                    new Translation('installer.php_version_not_found', array('version' => $version))
                )
            ),
            array(),
            new Translation('installer.php_version_is', array('version' => $version)),
            new Translation('installer.help_php_version')
        );
    }

    /**
     * Get the php version normalised.
     *
     * @return string
     */
    public function getPhpVersion()
    {
        $version = PHP_VERSION;
        $versionParser = new VersionParser;

        // Get the PHP version (normalise to remove any extra information).
        // From https://github.com/composer/composer/blob/28f0d8fcf7e2d6cb207b3c6ceca0e6142d141a7f/src/Composer/Repository/PlatformRepository.php#L77
        try {
            return $versionParser->normalize($version);
        } catch (UnexpectedValueException $e) {
            $prettyVersion = preg_replace('#^([^~+-]+).*$#', '$1', $version);

            return $versionParser->normalize($prettyVersion);
        }
    }
}
