<?php

namespace Supportpal\HelpdeskBaseRequirements;

use RuntimeException;
use SupportPal\AcceptLanguageParser\Parser;
use SupportPal\Requirements\Translation\Translation;
use SupportPal\Requirements\Translation\Translator as TranslatorInterface;

class Translator implements TranslatorInterface
{
    /**
     * Where to load languages from.
     *
     * @var string
     */
    private $directory;

    /**
     * List of available locales.
     *
     * @var string[]
     */
    private $locales;

    /**
     * Requirements language variables.
     *
     * @var array
     */
    private $LANG;

    /**
     * The locale to translate strings into.
     *
     * @var string
     */
    private $locale;

    /**
     * Fallback to this locale if the above does not exist.
     *
     * @var string
     */
    private $fallbackLocale = 'en';

    /**
     * Singleton Instance.
     *
     * @var static
     */
    private static $instance;

    /**
     * Translator constructor.
     *
     * @param string|null $directory
     */
    private function __construct($directory)
    {
        $this->directory = $directory;
    }

    /**
     * Singleton - get instance.
     *
     * @param string|null $directory
     * @return static
     */
    public static function getInstance($directory = null)
    {
        if (self::$instance === null) {
            if (! is_string($directory) || ! file_exists($directory)) {
                throw new RuntimeException('A valid directory is required to create a Translator instance.');
            }

            self::$instance = new static($directory);
        }

        return self::$instance;
    }

    /**
     * Set the locale to translate into.
     *
     * @param string $locale
     * @return static
     */
    public function setLocale($locale)
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * Get the locale to translate into.
     *
     * @return string
     */
    public function getLocale()
    {
        if ($this->locale === null) {
            return $this->fallbackLocale;
        }

        return $this->locale;
    }

    /**
     * Attempt to guess the user's locale.
     *
     * @return static
     */
    public function guessLocale()
    {
        if (isset($_GET['lang'])) {
            $this->locale = $_GET['lang'];

            return $this;
        }

        if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
            $parser = new Parser($_SERVER['HTTP_ACCEPT_LANGUAGE']);
            $locale = $parser->pick($this->availableLocalesInBcp47(), false);

            if ($locale !== null) {
                $this->locale = $locale->toIso15897();
            }
        }

        return $this;
    }

    /**
     * A naive approach to loading language variables, before Laravel has initialised.
     * Note: this function has no support for namespaces or plurals!
     *
     * @param Translation $translation
     * @return null|string
     */
    public function translate(Translation $translation)
    {
        // Get the filename and language key
        list($filename, $lang_key) = array_pad(explode('.', $translation->getKey()), 2, null);

        // Load the language
        $language = $this->getLanguageFile($this->getLocale(), $filename);
        if (is_array($language) && array_key_exists($lang_key, $language)) {
            // Get the language index
            $line = $language[$lang_key];

            // Perform replacements
            foreach ($translation->getParams() as $key => $value) {
                $line = str_replace(
                    array(':' . strtoupper($key), ':' . ucfirst($key), ':' . $key),
                    array(strtoupper($value), ucfirst($value), $value),
                    $line
                );
            }

            return $line;
        }

        return $translation->getKey();
    }

    /**
     * Get locale data from requested file.
     *
     * @param string $locale
     * @param string $filename
     * @return array|null
     */
    private function getLanguageFile($locale, $filename)
    {
        // Language has already been loaded so return it.
        if (isset($this->LANG[$locale][$filename])) {
            return $this->LANG[$locale][$filename];
        }

        // Try to load requested locale.
        $data = $this->loadLanguageFile($locale, $filename);
        if ($data !== null) {
            return $data;
        }

        // Requested locale does not exist, so try to load fallback locale.
        if ($locale !== $this->fallbackLocale) {
            return $this->loadLanguageFile($this->fallbackLocale, $filename);
        }

        return null;
    }

    /**
     * Get language file array.
     *
     * @param string $locale
     * @param string $filename
     * @return array|null
     */
    private function loadLanguageFile($locale, $filename)
    {
        // $locale can be supplied by the user (see guessLocale) so we need
        // to validate it exists in available locales to prevent path traversal.
        if (!in_array($locale, $this->availableLocales())) {
            return null;
        }

        $directories = $this->getLocaleDirectories($locale);

        // Check the language file exists
        foreach ($directories as $directory) {
            $lang_file = realpath(sprintf('%s/%s.php', $directory, $filename));
            if (!$lang_file || !file_exists($lang_file) || !is_readable($lang_file)) {
                continue;
            }

            // Load the language
            return $this->LANG[$locale][$filename] = require $lang_file;
        }

        return null;
    }

    /**
     * List of available locales.
     *
     * @return string[]
     */
    private function availableLocales()
    {
        if ($this->locales !== null) {
            return $this->locales;
        }

        return $this->locales = array_map(function ($e) {
            return basename($e);
        }, $this->getLocaleDirectories());
    }

    /**
     * List of available locales in BCP-47 format.
     *
     * @return string[]
     */
    private function availableLocalesInBcp47()
    {
        // convert supportpal locale to bcp-47
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Accept-Language#directives
        return array_map(function ($locale) {
            return str_replace('_', '-', $locale);
        }, $this->availableLocales());
    }

    /**
     * @param string $locale
     * @return array<int,string>
     */
    private function getLocaleDirectories($locale = '*')
    {
        $path = sprintf('%s/*/Lang/%s', $this->directory, $locale);

        $directories = glob($path, GLOB_ONLYDIR);

        if (!$directories) {
            return array();
        }

        return $directories;
    }
}
