<?php declare(strict_types=1);

namespace SupportPal\ProxyService;

use GuzzleHttp\Client;
use Illuminate\Contracts\Foundation\Application;
use SupportPal\ProxyService\Config\CurlConfig;
use SupportPal\ProxyService\Config\GuzzleConfig;
use SupportPal\ProxyService\Config\StreamConfig;

use function curl_setopt;

class ProxyService
{
    private ProxyConfig $config;

    private GuzzleConfig $guzzleConfig;

    private CurlConfig $curlConfig;

    private StreamConfig $streamConfig;

    private Application $app;

    public function __construct(
        ProxyConfig $config,
        GuzzleConfig $guzzleConfig,
        CurlConfig $curlConfig,
        StreamConfig $streamConfig,
        Application $app
    ) {
        $this->config = $config;
        $this->guzzleConfig = $guzzleConfig;
        $this->curlConfig = $curlConfig;
        $this->streamConfig = $streamConfig;
        $this->app = $app;
    }

    /**
     * Create a guzzle instance.
     *
     * @param mixed[] $options
     */
    public function createGuzzle(array $options = []): Client
    {
        $options = $options + $this->guzzleConfig->toArray();

        /** @var Client $client */
        $client = $this->app->make(Client::class, ['config' => $options]);

        return $client;
    }

    /**
     * Set proxy parameters for a CURL connection.
     *
     * @param  resource $resource
     */
    public function curl(&$resource): void
    {
        foreach ($this->curlOpts() as $option => $value) {
            curl_setopt($resource, $option, $value);
        }
    }

    /**
     * Get curl proxy options.
     *
     * @return mixed[]
     */
    public function curlOpts(): array
    {
        return $this->curlConfig->toArray();
    }

    /**
     * stream_context_create configuration options.
     * https://www.php.net/manual/en/context.http.php
     *
     * @return mixed[]
     */
    public function streamOpts(): array
    {
        return $this->streamConfig->toArray();
    }

    /**
     * Get proxy URI.
     *
     * @return mixed
     */
    public function getUri()
    {
        return $this->config->getUri();
    }

    /**
     * Check whether a proxy is set.
     */
    public function isset(): bool
    {
        return $this->config->hasUri();
    }
}
