<?php

namespace SupportPal\Requirements;

use SupportPal\Requirements\Translation\Translation;

class RequirementGroup
{
    /**
     * Name of the group.
     *
     * @var Translation
     */
    private $name;

    /**
     * Description of the requirements group.
     *
     * @var Translation
     */
    private $description;

    /**
     * Message to be displayed.
     *
     * @var Translation
     */
    private $message;

    /**
     * List of required requirements.
     *
     * @var Requirement[]
     */
    private $required;

    /**
     * List of optional requirements.
     *
     * @var Requirement[]
     */
    private $optional;

    /**
     * Turn off caching of requirements.
     *
     * @var bool
     */
    private $force = false;

    /**
     * Initialise the requirements group
     *
     * @param  Translation          $name
     * @param  array                $required
     * @param  array                $optional
     * @param  Translation|callable $message        The callback should return a Translation instance.
     * @param  Translation          $description
     */
    public function __construct(
        Translation $name,
        array $required = array(),
        array $optional = array(),
        $message = null,
        Translation $description = null
    ) {
        $this->name = $name;
        $this->description = $description;
        $this->message = $message;
        $this->required = $required;
        $this->optional = $optional;
    }

    /**
     * Turn off caching of requirements.
     *
     * @param  bool $value
     * @return $this
     */
    public function cache($value)
    {
        $this->force = $value;

        return $this;
    }

    /**
     * Check whether all requirements are met.
     *
     * @return bool
     */
    public function allValid()
    {
        return $this->isValid(array_merge($this->required, $this->optional));
    }

    /**
     * Check whether the required requirements are met.
     *
     * @return bool
     */
    public function requiredValid()
    {
        return $this->isValid($this->required);
    }

    /**
     * Check whether the optional requirements are valid.
     *
     * @return bool
     */
    public function optionalValid()
    {
        return $this->isValid($this->optional);
    }

    /**
     * Count total number of requirements that are valid.
     *
     * @param  array $requirements
     * @return int
     */
    public function countValid(array $requirements)
    {
        $count = 0;

        foreach ($requirements as $requirement) {
            if ($requirement->isValid()) {
                $count++;
            }
        }

        return $count;
    }

    /**
     * Get the group name
     *
     * @return Translation
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Get the description of the requirements group.
     *
     * @return Translation
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Get list of required requirements.
     *
     * @return Requirement[]
     */
    public function getRequired()
    {
        return $this->required;
    }

    /**
     * Get list of optional requirements.
     *
     * @return Requirement[]
     */
    public function getOptional()
    {
        return $this->optional;
    }

    /**
     * Get the original message.
     *
     * @return callable|null|Translation
     */
    public function getRawMessage()
    {
        return $this->message;
    }

    /**
     * Add a new required requirement to the group.
     *
     * @param  Requirement $requirement
     * @return void
     */
    public function addRequiredRequirement(Requirement $requirement)
    {
        $this->required[] = $requirement;
    }

    /**
     * Add a new optional requirement to the group.
     *
     * @param  Requirement $requirement
     * @return void
     */
    public function addOptionalRequirement(Requirement $requirement)
    {
        $this->optional[] = $requirement;
    }

    /**
     * Get the message to be displayed to the user.
     *
     * @return Translation
     */
    public function getMessage()
    {
        if (is_callable($this->message)) {
            return call_user_func_array($this->message, array($this));
        } else if (count($this->required) && ! count($this->optional)) {
            // Only required requirements.
            return new Translation(
                "installer.required_requirements",
                array("required" => $this->countValid($this->required), "total" => count($this->required))
            );
        } else if (! count($this->required) && count($this->optional)) {
            // Only optional requirements.
            return new Translation(
                "installer.optional_requirements",
                array("optional" => $this->countValid($this->optional), "total" => count($this->optional))
            );
        } else {
            // Both required and optional requirements.
            return new Translation(
                "installer.both_requirements",
                array(
                    "required"       => $this->countValid($this->required),
                    "total_required" => count($this->required),
                    "optional"       => $this->countValid($this->optional),
                    "total_optional" => count($this->optional),
                )
            );
        }
    }

    /**
     * Check with an array of requirements is met.
     *
     * @param  array $requirements
     * @return bool
     */
    private function isValid(array $requirements)
    {
        foreach ($requirements as $requirement) {
            if (! $requirement->isValid($this->force)) {
                return false;
            }
        }

        return true;
    }
}
