<?php

namespace SupportPal\Requirements;

use InvalidArgumentException;
use SupportPal\Requirements\Translation\Translation;
use SupportPal\Requirements\Translation\Translator;

class Requirements
{
    /**
     * Singleton Instance.
     *
     * @var static
     */
    private static $instance;

    /**
     * List of requirement groups.
     *
     * @var RequirementGroup[]
     */
    protected $requirements;

    /** @var Translator */
    private $translator;

    /**
     * Initialise all of the system requirements.
     *
     * @param  Translator $translator
     * @param  RequirementGroup[] $requirements
     */
    private function __construct(Translator $translator, array $requirements)
    {
        $this->translator = $translator;
        $this->requirements = $requirements;
    }

    /**
     * Singleton - get instance.
     *
     * @param  Translator $translator
     * @param  RequirementGroup[] $requirements
     * @return static
     */
    public static function getInstance($translator = null, array $requirements = array())
    {
        if (! empty($translator) && ! empty($requirements)) {
            self::$instance = new self($translator, $requirements);
        }

        return self::$instance;
    }

    /**
     * All requirements.
     *
     * @param  bool $force Whether to ignore the cache or not.
     * @return RequirementGroup[]
     */
    public function all($force = false)
    {
        $requirements = $this->requirements;
        foreach ($requirements as &$group) {
            $group = $group->cache($force);
        }

        return $this->requirements;
    }

    /**
     * Get all required requirements.
     *
     * @return RequirementGroup[]
     */
    public function required()
    {
        $required = array();

        foreach ($this->requirements as $group) {
            $required[] = new RequirementGroup(
                $group->getName(),
                $group->getRequired(),
                array(),
                $group->getRawMessage(),
                $group->getDescription()
            );
        }

        return $required;
    }

    /**
     * Get all optional requirement groups.
     *
     * @return RequirementGroup[]
     */
    public function optional()
    {
        $optional = array();

        foreach ($this->requirements as $group) {
            $optional[] = new RequirementGroup(
                $group->getName(),
                array(),
                $group->getOptional(),
                $group->getRawMessage(),
                $group->getDescription()
            );
        }

        return $optional;
    }

    /**
     * Check whether's all of the compulsory system requirements are met
     *
     * @return bool
     */
    public function valid()
    {
        foreach ($this->requirements as $group) {
            if (! $group->requiredValid()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Add new requirement groups.
     *
     * @param  RequirementGroup[] $requirements
     * @return void
     */
    public function add(array $requirements)
    {
        $this->requirements = array_merge($this->requirements, $requirements);
    }

    /**
     * Prepend a requirement to the front of the list.
     *
     * @param  RequirementGroup $requirement
     * @return void
     */
    public function prepend(RequirementGroup $requirement)
    {
        array_unshift($this->requirements, $requirement);
    }

    /**
     * Get a given requirement group.
     *
     * @param  mixed $key
     * @return RequirementGroup
     * @throws InvalidArgumentException
     */
    public function get($key)
    {
        if (! isset($this->requirements[$key])) {
            throw new InvalidArgumentException("No such key: '$key'.");
        }

        return $this->requirements[$key];
    }

    /**
     * Set a requirement group.
     *
     * @param  mixed            $key
     * @param  RequirementGroup $requirement
     * @return void
     */
    public function set($key, RequirementGroup $requirement)
    {
        $this->requirements[$key] = $requirement;
    }

    /**
     * Render requirements error string for console.
     *
     * @param  bool   $force  Whether to ignore the cache or not.
     * @return void
     */
    public function renderForConsole($force = false)
    {
        echo $this->translator->translate(new Translation('installer.requirements_failed')) . "\n\n";
        foreach ($this->all($force) as $requirementGroup) {
            if ($requirementGroup->requiredValid()) {
                continue;
            }

            echo "[{$this->translator->translate($requirementGroup->getName())}]\n";

            foreach ($requirementGroup->getRequired() as $requirement) {
                if (! $requirement->isValid()) {
                    echo "\t{$this->translator->translate($requirement->getName())}"
                        . " - {$this->translator->translate($requirement->getFailureMsg())}\n";
                }
            }
        }
    }
}
