<?php declare(strict_types=1);

namespace SupportPal\Support\Archive;

use SupportPal\Support\Exception\ArchiveTypeNotSupportedException;
use Symfony\Component\Mime\MimeTypes;

class ArchiveManagerCollection implements ArchiveManager
{
    /** @var ArchiveManager[] */
    private array $archiveManagers;

    private MimeTypes $mimeTypes;

    /**
     * ArchiveManagerCollection constructor.
     * @param ArchiveManager[] $archiveManagers
     */
    public function __construct(array $archiveManagers, MimeTypes $mimeTypes)
    {
        $this->archiveManagers = $archiveManagers;
        $this->mimeTypes = $mimeTypes;
    }

    /**
     * @throws ArchiveTypeNotSupportedException
     */
    public function extract(string $source, string $target, bool $isCompressed): void
    {
        $manager = $this->getSupportingManagerOrFail($source);

        $manager->extract($source, $target, $isCompressed);
    }

    public function supports(string $mimeType): bool
    {
        return $this->getSupportingManager($mimeType) !== null;
    }

    private function getSupportingManager(string $mimeType): ?ArchiveManager
    {
        foreach ($this->archiveManagers as $archiveManager) {
            if ($archiveManager->supports($mimeType)) {
                return $archiveManager;
            }
        }

        return null;
    }

    /**
     * @param string $source
     * @return ArchiveManager
     */
    private function getSupportingManagerOrFail(string $source): ArchiveManager
    {
        $mimeType = (string) $this->mimeTypes->guessMimeType($source);
        $manager = $this->getSupportingManager($mimeType);
        if ($manager === null) {
            throw new ArchiveTypeNotSupportedException;
        }

        return $manager;
    }
}
