<?php declare(strict_types=1);

namespace SupportPal\Support\Bootstrap;

use Illuminate\Config\Repository;
use Illuminate\Contracts\Config\Repository as RepositoryContract;
use Illuminate\Foundation\Application;
use SplFileInfo;
use Symfony\Component\Finder\Finder;

use function array_replace_recursive;
use function basename;
use function date_default_timezone_set;
use function file_exists;
use function is_dir;
use function ksort;
use function mb_internal_encoding;
use function realpath;
use function str_replace;
use function trim;

use const DIRECTORY_SEPARATOR;
use const SORT_NATURAL;

class LoadConfiguration
{
    /**
     * You can use patterns (delimited with / sign), globs or simple strings.
     *
     *     $finder->name('*.php')
     *     $finder->name('/\.php$/') // same as above
     *     $finder->name('test.php')
     *     $finder->name(['test.py', 'test.php'])
     *
     * @var string|string[]
     */
    protected $pattern = '*.php';

    protected Application $app;

    /**
     * Bootstrap the given application.
     */
    public function bootstrap(Application $app): void
    {
        $this->app = $app;

        $config = $this->bindConfigRepository();

        $this->loadConfigurationFiles($config);

        date_default_timezone_set($config->get('app.timezone', 'UTC'));

        mb_internal_encoding('UTF-8');
    }

    protected function bindConfigRepository(): RepositoryContract
    {
        $abstract = 'config';
        if ($this->app->bound($abstract)) {
            return $this->app->make('config');
        }

        $this->app->instance($abstract, $config = new Repository([]));

        return $config;
    }

    protected function loadConfigurationFiles(RepositoryContract $repository): void
    {
        foreach ($this->getConfigurationFiles($this->getConfigPath()) as $key => $path) {
            $repository->set($key, require $path);
        }

        $env = $this->detectEnvironment($repository);

        $this->loadEnvConfigurationFiles($repository, $env);
    }

    protected function detectEnvironment(RepositoryContract $config): string
    {
        if ($this->app->bound('env')) {
            /** @var string $env */
            $env = $this->app->environment();

            return $env;
        }

        return $this->app->detectEnvironment(function () use ($config) {
            return $config->get('app.env', 'production');
        });
    }

    protected function loadEnvConfigurationFiles(RepositoryContract $repository, string $env): void
    {
        $envConfigPath = (string) realpath($this->getConfigPath() . '/' . $env);
        if (! file_exists($envConfigPath) || ! is_dir($envConfigPath)) {
            return;
        }

        foreach ($this->getConfigurationFiles($envConfigPath) as $key => $path) {
            $oldValues = $repository->get($key) ?: [];
            $newValues = require $path;

            // Replace any matching values in the old config with the new ones.
            $repository->set($key, array_replace_recursive($oldValues, $newValues));
        }
    }

    /**
     * @return string[]
     */
    protected function getConfigurationFiles(string $path): array
    {
        $files = [];

        foreach (Finder::create()->files()->depth(0)->name($this->pattern)->in($path) as $file) {
            $directory = $this->getNestedDirectory($file, $path);

            $files[$directory.basename($file->getRealPath(), '.php')] = $file->getRealPath();
        }

        ksort($files, SORT_NATURAL);

        return $files;
    }

    /**
     * Get the configuration file nesting path.
     */
    protected function getNestedDirectory(SplFileInfo $file, string $configPath): string
    {
        $directory = $file->getPath();

        if ($nested = trim(str_replace($configPath, '', $directory), DIRECTORY_SEPARATOR)) {
            $nested = str_replace(DIRECTORY_SEPARATOR, '.', $nested).'.';
        }

        return $nested;
    }

    protected function getConfigPath(): string
    {
        return $this->app->configPath();
    }
}
