<?php declare(strict_types=1);

namespace SupportPal\Support\Database;

use Closure;
use Exception;
use Illuminate\Database\Connection as BaseConnection;
use Illuminate\Support\Facades\Cache;
use PDO;
use PDOException;
use stdClass;
use Throwable;

use function array_column;
use function now;
use function preg_match;
use function rtrim;
use function sprintf;
use function stripos;
use function version_compare;

class Connection extends BaseConnection
{
    /**
     * Check MySQL server version.
     */
    public function meetsServerVersion(string $given): bool
    {
        try {
            $version = $this->getServerVersion();

            return version_compare($version, $given, '>=');
        } catch (PDOException $e) {
            return false;
        }
    }

    /**
     * Get MySQL server version.
     *
     * @return string
     */
    public function getServerVersion()
    {
        $query = $this->select('SELECT @@version as version');

        return array_column($query, 'version')[0] ?? '';
    }

    /**
     * Check MySQL client version.
     */
    public function meetsClientVersion(string $givenClient, string $givenServer): bool
    {
        try {
            $version = $this->getClientVersion();
            preg_match('/((?:[0-9]{1,2}\.){1,3}[0-9]{1,2})/', $version, $matches);
            if (! isset($matches[1])) {
                return false;
            }

            // Check client version - utf8mb4 requires mysqlnd 5.0.9+
            if (stripos($version, 'mysqlnd') !== false) {
                return version_compare($matches[1], $givenClient, '>=');
            }

            // Fallback to server version requirement.
            return version_compare($matches[1], $givenServer, '>=');
        } catch (PDOException $e) {
            // In case of the driver doesn't support getting attributes
            return false;
        }
    }

    /**
     * Get MySQL client version.
     *
     * @return string
     */
    public function getClientVersion()
    {
        return $this->getPdo()->getAttribute(PDO::ATTR_CLIENT_VERSION);
    }

    /**
     * Get the max allowed packet size for the MySQL Server in Kilobytes.
     *
     * @return int
     */
    public function maxPacketSize(): int
    {
        return (int) Cache::remember('max_allowed_packet', now()->addMinutes(86400), function () {
            $query = $this->select('SELECT @@max_allowed_packet as max_allowed_packet');

            $max = array_column($query, 'max_allowed_packet')[0] ?? 1048576;

            return $max / 1024; // Convert bytes to KB
        });
    }

    /**
     * Get all table columns that are not using the utf8mb4_unicode_ci collation.
     *
     * @return mixed[]
     */
    public function getNonUtf8mb4CollationColumns(): array
    {
        $query = 'SELECT * FROM information_schema.COLUMNS WHERE table_schema = ? AND collation_name != ?';

        return $this->select($query, [$this->getDatabaseName(), 'utf8mb4_unicode_ci']);
    }

    /**
     * Get all tables that are not using the utf8mb4_unicode_ci collation.
     *
     * @return mixed[]
     */
    public function getNonUtf8mb4CollationTables(): array
    {
        $query = 'SELECT * FROM information_schema.TABLES WHERE table_schema = ? AND table_collation != ?';

        return $this->select($query, [$this->getDatabaseName(), 'utf8mb4_unicode_ci']);
    }

    /**
     * Get all table columns that are not using the utf8mb4 character set.
     *
     * @return mixed[]
     */
    public function getNonUtf8mb4CharsetColumns(): array
    {
        $query = 'SELECT * FROM information_schema.COLUMNS WHERE table_schema = ? AND character_set_name != ?';

        return $this->select($query, [$this->getDatabaseName(), 'utf8mb4']);
    }

    /**
     * Convert information_schema output to a comma delimited list (use in combination with
     * getNonUtf8mb4CharsetColumns, getNonUtf8mb4CollationTables, getNonUtf8mb4CollationColumns).
     *
     * @param  stdClass[] $records
     * @return string
     */
    public function schemaToString(array $records)
    {
        $string = '';
        foreach ($records as $record) {
            if (isset($record->COLUMN_NAME)) {
                $string .= sprintf('%s.%s, ', $record->TABLE_NAME, $record->COLUMN_NAME);
            } else {
                $string .= sprintf('%s, ', $record->TABLE_NAME);
            }
        }

        return rtrim($string, ', ');
    }

    /**
     * Get list of InnoDB full text stop words.
     *
     * @return mixed[]
     */
    public function getStopWords(): array
    {
        return (array) Cache::rememberForever('fulltext_stopwords', function () {
            $query = $this->select('SELECT * FROM INFORMATION_SCHEMA.INNODB_FT_DEFAULT_STOPWORD');

            return array_column($query, 'value');
        });
    }

    /**
     * Get the InnoDB full text minimum token size.
     *
     * @return int
     */
    public function getFtsMinTokenSize(): int
    {
        return (int) Cache::rememberForever('fulltext_min_token_size', function () {
            $size = $this->select('SELECT @@innodb_ft_min_token_size AS size');

            return array_column($size, 'size')[0] ?? 0;
        });
    }

    /**
     * Execute a Closure within a transaction.
     *
     * @param  Closure $callback
     * @param  int     $attempts Default to 5 attempts.
     * @return mixed
     *
     * @throws Exception|Throwable
     */
    public function transaction(Closure $callback, $attempts = 5)
    {
        return parent::transaction($callback, $attempts);
    }
}
