<?php declare(strict_types=1);

namespace SupportPal\Support\Files;

use SupportPal\Support\Exception\BinaryNotFoundException;

use function array_filter;
use function count;
use function explode;
use function getenv;
use function implode;
use function ini_get;
use function sprintf;

use const PATH_SEPARATOR;

trait ExecutableFinderConcerns
{
    private function binaryNotFoundException(string $binaryName): BinaryNotFoundException
    {
        $message = sprintf('Unable to find the %s binary', $binaryName);
        if (ini_get('open_basedir')) {
            $dirs = $this->getOpenBaseDirs();
            $message .= sprintf(', within (%s). open_basedir restriction in effect.', implode(', ', $dirs));
        } else {
            $dirs = $this->getPathDirs();
            if (count($dirs) === 0) {
                $message .= '. The PATH/Path environment variable is not set.';
            } else {
                $message .= sprintf(', within (%s).', implode(', ', $dirs));
            }
        }

        return new BinaryNotFoundException($message);
    }

    /**
     * @return string[]
     */
    private function getOpenBaseDirs(): array
    {
        $paths = ini_get('open_basedir');
        if ($paths === false) {
            $paths = '';
        }

        return explode(PATH_SEPARATOR, $paths);
    }

    /**
     * @return string[]
     */
    private function getPathDirs(): array
    {
        $paths = getenv('PATH') ?: getenv('Path');
        if ($paths === false) {
            $paths = '';
        }

        return array_filter(explode(PATH_SEPARATOR, $paths));
    }
}
