<?php declare(strict_types=1);

namespace SupportPal\Support\Logging;

use Carbon\Carbon;
use SplObjectStorage;
use SplObserver;
use SplSubject;

use function sprintf;

class Logger implements SplSubject
{
    /**
     * Observers.
     *
     * @var SplObjectStorage<SplObserver>
     */
    protected SplObjectStorage $observers;

    protected string $message = '';

    /**
     * UpdateLog constructor.
     */
    public function __construct()
    {
        $this->observers = new SplObjectStorage;
    }

    /**
     * Notify observers of a new log entry.
     *
     * @param string $message
     * @return void
     */
    public function log(string $message): void
    {
        $this->message = sprintf('[%s] %s', Carbon::now()->toDate()->format('Y-m-d H:i:s'), $message);

        $this->notify();
    }

    /**
     * Fetch the latest log entry.
     *
     * @return string
     */
    public function message(): string
    {
        return $this->message;
    }

    /**
     * Attach an SplObserver
     * @link https://php.net/manual/en/splsubject.attach.php
     * @param SplObserver $observer <p>
     * The <b>SplObserver</b> to attach.
     * </p>
     * @return void
     */
    public function attach(SplObserver $observer): void
    {
        $this->observers->attach($observer);
    }

    /**
     * Detach an observer
     * @link https://php.net/manual/en/splsubject.detach.php
     * @param SplObserver $observer <p>
     * The <b>SplObserver</b> to detach.
     * </p>
     * @return void
     */
    public function detach(SplObserver $observer): void
    {
        $this->observers->detach($observer);
    }

    /**
     * Notify an observer
     * @link https://php.net/manual/en/splsubject.notify.php
     * @return void
     */
    public function notify(): void
    {
        foreach ($this->observers as $observer) {
            $observer->update($this);
        }
    }
}
