<?php declare(strict_types=1);

namespace SupportPal\Support\Str;

use function is_string;
use function phpversion;
use function preg_replace;
use function str_replace;
use function trim;
use function version_compare;

class Trim
{
    /**
     * Remove erroneous white space (this includes the unicode NULL character (x00) and also zero-width space
     * characters):
     *   - U+200B zero width space
     *   - U+200C zero width non-joiner Unicode code point
     *   - U+200D zero width joiner Unicode code point
     *   - U+FEFF zero width no-break space Unicode code point
     *
     * @see http://stackoverflow.com/questions/11305797/remove-zero-width-space-characters-from-a-javascript-string
     */
    public function trim(): callable
    {
        return function ($string): string {
            if (version_compare(phpversion(), '7.0', '<')) {
                $clean = preg_replace('/[\\x{0000}\\x{200B}-\\x{200D}\\x{FEFF}]|\\r?\\n|\\r/u', '', $string);
            } else {
                // In PHP 7.0, support for Unicode special character sequence was added
                // http://ch.php.net/manual/en/language.types.string.php#language.types.string.syntax.double
                //
                // Due to a bug in Centos PCRE implementation (https://bugzilla.redhat.com/show_bug.cgi?id=1400267)
                // we have had to add this special case using str_replace instead of preg_replace.

                $search = ["\u{0000}", "\u{200B}", "\u{200C}", "\u{200D}", "\u{FEFF}", "\r\n", "\r", "\n"];

                $clean = str_replace($search, '', $string);
            }

            // Remove unnecessary ascii whitespace between words.
            $clean = preg_replace('/\s\s+/', ' ', $clean);

            return is_string($clean) ? trim($clean) : $string;
        };
    }
}
