<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Selfservice\Models\FeedbackLog;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.user_feedback');
$report->description = Lang::get('report.user_feedback_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'custom';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = null;

// Setting out the initial query for the report data before any filtering is applied
$report->query = FeedbackLog::whereNotNull('rating')
    ->with('article', 'user')
    ->orderBy('updated_at', 'DESC');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Filter by article ID.
    if (request()->has('article_id')) {
        $query->where('article_id', request()->input('article_id'));
    }

    // Set the start and end time
    if ($report->startDate > 0) {
        $query->where("created_at", '>=', $report->startDate);
    }
    if ($report->endDate > 0) {
        $query->where("created_at", '<=', $report->endDate);
    }

    // Execute the query
    $result = $query->get();

    // Start building returning data array
    $data = [];

    // Add the column headers
    $data[] = [
        Lang::get('general.date'),
        Lang::choice('selfservice.article', 1),
        Lang::choice('user.user', 1),
        Lang::get('ticket.rating'),
        ''
    ];

    foreach ($result as $log) {
        // Article URL
        $articleRoute = route('selfservice.operator.article.edit', [$log->article->id]);

        // User URL.
        $user = $log->user_ip;
        if (isset($log->user)) {
            $userRoute = route('user.operator.user.edit', [$log->user->id]);
            $user = "<img src='{$log->user->avatar_url}' class='sp-avatar sp-max-w-2xs'>" . "&nbsp; <a href='{$userRoute}'>" . e($log->user->formatted_name) . "</a>";
        }

        // Feedback rating.
        $rating = Lang::get('general.unknown');
        if ($log->rating == FeedbackLog::BAD) {
            $rating = Lang::get('general.bad');
        } elseif ($log->rating == FeedbackLog::NEUTRAL) {
            $rating = Lang::get('general.neutral');
        } elseif ($log->rating == FeedbackLog::GOOD) {
            $rating = Lang::get('general.good');
        }

        // View Feedback button.
        $button = null;
        if ($log->fields_answered) {
            $url = route('selfservice.operator.feedback.show', [$log->id]);
            $button = Form::button(
                Lang::get('ticket.view_feedback'),
                [
                    'class' => 'view-feedback sp-button-sm sp-float-right sp-whitespace-nowrap',
                    'onclick' => "window.open('{$url}','width=800,height=600,scrollbars=yes');"
                ]
            )->toHtml();
        }

        $data[] = [
            formatDate($log->updated_at),
            "<a href='{$articleRoute}'>" . e(Str::limit($log->article->title, 50)) . "</a>",
            $user,
            $rating,
            $button
        ];
    }

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
