<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Ticket\Models\Message;
use App\Modules\Ticket\Models\Ticket;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.busiest_month');
$report->description = Lang::get('report.busiest_month_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'yearly';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['line', 'table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = 'ticket';

// Setting out the initial query for the report data before any filtering is applied
$report->query = Ticket::leftJoin('ticket_message', 'ticket.id', '=', 'ticket_message.ticket_id')
    ->select([
        DB::raw('count(DISTINCT ticket_message.id) as count'),
        // Need to apply timezone offset here to as we're using from_unixtime function
        DB::raw('month(from_unixtime(`ticket_message`.`created_at` - ' . timeWithOffset(0) . ')) as month')
    ])
    ->where('ticket_message.by', Message::USER)
    ->groupBy('month')
    ->orderBy('month', 'asc');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Set the start and end time
    if ($report->startDate > 0) {
        $query = $query->where('ticket_message.created_at', '>=', $report->startDate);
    }
    if ($report->endDate > 0) {
        $query = $query->where('ticket_message.created_at', '<=', $report->endDate);
    }

    // Start building returning data array
    $data[] = [Lang::choice('general.month', 1), Lang::get('general.count')];
    $data[] = [Lang::get('general.january'), 0];
    $data[] = [Lang::get('general.february'), 0];
    $data[] = [Lang::get('general.march'), 0];
    $data[] = [Lang::get('general.april'), 0];
    $data[] = [Lang::get('general.may'), 0];
    $data[] = [Lang::get('general.june'), 0];
    $data[] = [Lang::get('general.july'), 0];
    $data[] = [Lang::get('general.august'), 0];
    $data[] = [Lang::get('general.september'), 0];
    $data[] = [Lang::get('general.october'), 0];
    $data[] = [Lang::get('general.november'), 0];
    $data[] = [Lang::get('general.december'), 0];

    // Loop over the results.
    foreach ($query->get() as $row) {
        $data[$row['month']][1] = (int) $row->count;
    }

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
