<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Ticket\Models\CannedResponse;
use App\Modules\Ticket\Models\CannedResponseUsage;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.canned_response_usage');
$report->description = Lang::get('report.canned_response_usage_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'none';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = null;

// Setting out the initial query for the report data before any filtering is applied
$report->query = CannedResponseUsage::select(['response_id', DB::raw('SUM(`times`) as count')])
    ->groupBy('response_id')
    ->orderBy('count', 'desc');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Start building returning data array
    $data[] = [Lang::choice('ticket.cannedresponse', 1), Lang::get('general.count')];

    // Fetch initial data - so we can show zero for any values that don't come up in the real data
    $responses = CannedResponse::get(['id', 'name'])->pluck('name', 'id');

    // Store the counts.
    foreach ($query->get()->toArray() as $value) {
        $data[$value['response_id']] = [e($responses[$value['response_id']]), (int) $value['count']];
    }

    // Fill 0s for missing responses.
    foreach ($responses as $id => $name) {
        if (! isset($data[$id])) {
            $data[$id] = [e($name), 0];
        }
    }

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
