<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Ticket\Models\Message;
use App\Modules\Ticket\Models\Ticket;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.first_response_time');
$report->description = Lang::get('report.first_response_time_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'custom';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['bar', 'pie', 'table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = 'ticket';

// Setting out the initial query for the report data before any filtering is applied
$report->query = Ticket::select(['ticket.id'])
    ->has('messages', '>', 1)
    ->with(['messages' => function ($query) {
        $query->select('ticket_id', 'by', 'type', 'created_at')->orderBy('created_at');
    }])
    ->groupBy('ticket.id');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Set the start and end time
    if ($report->startDate > 0) {
        $query = $query->where('ticket.created_at', '>=', $report->startDate);
    }
    if ($report->endDate > 0) {
        $query = $query->where('ticket.created_at', '<=', $report->endDate);
    }

    // Execute the query
    $tickets = $query->get();

    // Start building returning data array
    $data = [];

    // Add the column headers
    $data[] = [Lang::get('report.first_response_time'), Lang::get('report.ticket_count')];

    // Initialise the new result array, created by the below algorithm
    $result = [];

    // Loop over all the tickets
    foreach ($tickets as $ticket) {
        // Get the messages for this ticket
        $messages = $ticket->messages;

        // Get the first user message
        $userKey = $messages->search(function ($item, $key) {
            return $item->by === Message::USER;
        });
        if ($userKey === false) {
            continue; // The ticket doesn't contain a user message
        }
        $firstUserMsg = $messages->get($userKey);

        // Get the first operator message
        $operatorKey = $messages->search(function ($item, $key) use ($userKey) {
            return $item->by === Message::OPERATOR && $item->type === Message::TYPE_MESSAGE && $key > $userKey;
        });
        if ($operatorKey === false) {
            continue; // The ticket hasn't been replied to by an operator
        }
        $firstOperatorMsg = $messages->get($operatorKey);

        // Calculate the difference between the first message and first operator reply
        // Ths returns the time difference in hours
        $result[] = [
            'first_reply_time' => intval(floor(($firstOperatorMsg->created_at - $firstUserMsg->created_at) / 3600))
        ];
    }


    // To build up the data in predefined periods
    $hours = [0, 1, 3, 6, 12, 24];
    $hourCounts = [
        1  => 0,
        3  => 0,
        6  => 0,
        12 => 0,
        24 => 0,
    ];

    // The count for anything above the cutoff of 24
    $cutoffCount = 0;

    // Add each row of data, ensure it matches the column header
    if (is_array($result)) {
        foreach ($result as $value) {
            // Use this to store when we've found what group it fits in
            $found = false;

            foreach ($hours as $key => $hour) {
                // Is it within this hour and the next period start
                if ($value['first_reply_time'] >= $hour && isset($hours[$key + 1])
                    && ($value['first_reply_time'] < $hours[$key + 1])
                ) {
                    // Add to the count of this group and declare it found
                    $hourCounts[$hours[$key + 1]]++;
                    $found = true;
                    // Don't need to run this foreach any more
                    break;
                }
                // Not in this group, carry on
            }

            // Not found in the groups above, add it to 24+
            if (! $found) {
                $cutoffCount++;
            }
        }
    }

    // Add the rows of data
    $data[] = [Lang::get('report.lt_1_hour'), $hourCounts[1]];
    $data[] = [Lang::get('report.1-3_hour'), $hourCounts[3]];
    $data[] = [Lang::get('report.3-6_hour'), $hourCounts[6]];
    $data[] = [Lang::get('report.6-12_hour'), $hourCounts[12]];
    $data[] = [Lang::get('report.12-24_hour'), $hourCounts[24]];
    $data[] = [Lang::get('report.24_plus_hour'), $cutoffCount];

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
