<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Ticket\Models\Message;
use App\Modules\Ticket\Models\Ticket;
use App\Modules\User\Models\User;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.first_response_by_operator');
$report->description = Lang::get('report.first_response_by_operator_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'custom';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['bar', 'pie', 'table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = 'ticket';

// Setting out the initial query for the report data before any filtering is applied
$report->query = Ticket::select(['ticket.id'])
    ->has('messages', '>', 1)
    ->with(['messages' => function ($query) {
        $query->select('ticket_id', 'user_id', 'by', 'type', 'created_at')->orderBy('created_at');
    }])
    ->groupBy('ticket.id');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Set the start and end time
    if ($report->startDate > 0) {
        $query = $query->where('ticket.created_at', '>=', $report->startDate);
    }
    if ($report->endDate > 0) {
        $query = $query->where('ticket.created_at', '<=', $report->endDate);
    }

    // Execute the query
    $tickets = $query->get();

    // Start building returning data array
    $data = [];

    // Add the column headers
    $data[] = [Lang::choice('general.operator', 1), 'Number of hours'];

    // Fetch initial data - so we can show zero for any values that don't come up in the real data
    $initialData = User::operator()->get(['id', 'role', 'firstname', 'lastname', 'email']);

    // Set value 0 for each one
    foreach ($initialData as $row) {
        $data[$row['id']] = [e($row['formatted_name']), 0];
    }

    // Loop over all the tickets
    $result = [];
    foreach ($tickets as $ticket) {
        // Get the messages for this ticket
        $messages = $ticket->messages;

        // Get the first user message
        $userKey = $messages->search(function ($item, $key) {
            return $item->by === Message::USER;
        });
        if ($userKey === false) {
            continue; // The ticket doesn't contain a user message
        }
        $firstUserMsg = $messages->get($userKey);

        // Get the first operator message
        $operatorKey = $messages->search(function ($item, $key) use ($userKey) {
            return $item->by === Message::OPERATOR && $item->type === Message::TYPE_MESSAGE && $key > $userKey;
        });
        if ($operatorKey === false) {
            continue; // The ticket hasn't been replied to by an operator
        }
        $firstOperatorMsg = $messages->get($operatorKey);

        // Calculate the difference between the first message and first operator reply
        // This returns the time difference in hours
        $result[] = [
            'operator_id'      => $firstOperatorMsg->user_id,
            'first_reply_time' => ($firstOperatorMsg->created_at - $firstUserMsg->created_at) / 3600
        ];
    }

    // Split the results per operator
    $result = collect($result)->groupBy('operator_id');

    // Get an average and set it in the report data
    foreach ($result as $id => $time) {
        $data[$id][1] = round($time->average('first_reply_time'), 2);
    }

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
