<?php
/**
 * SupportPal Report File
 *
 * Quick Guide:
 * 1. Generate the required data by updating $report->query in all instances
 * 2. Update column headers in $data and then iterate through report data to add each row
 * 3. Update the 5 other variables with the information/options you require
 * 4. Save the file like NewReportName.php if you intend it to show as "New Report Name"
 */

// Use any models needed for data
use App\Modules\Ticket\Models\Ticket;

$report = new stdClass;

// Give the report a name and description
$report->title = Lang::get('report.time_to_resolution');
$report->description = Lang::get('report.time_to_resolution_desc');

// The report timeframe, possible options:
// monthly - allows selection of data for report for each month
// yearly  - allows selection of data for report for each year
// custom  - allows to choose any window of time for the report as well as all time, defaults to one month
//           from current date but can be overridden by setting $startDate and $endDate (UNIX timestamps)
// none    - for reports where time is not needed
$report->timeframe = 'custom';

// The report chart options, an array with the following possible values, with the first value being shown by default:
// pie       - Pie chart, requires just two columns of data
// bar       - Bar chart, works with multiple columns of data
// column    - Column chart, works with multiple columns of data
// histogram - Histogram, requires just two columns of data
// line      - Line graph, requires three columns of data where one column is time-based
// area      - Area graph, requires three columns of data where one column is time-based
// stacked   - Stacked Area graph, similar to area graph but the various rows are stacked on top of each other
// stepped   - Stepped Area graph, similar to area graph but the various rows are stacked on top of each other
// geo       - Geo graph, requires just two columns of data where one is geographical-based
// table     - Table, visible on another tab
$report->charts = ['bar', 'pie', 'table'];

// The report filtering options, either 'ticket' or 'user', or can be disabled by setting to null.
// ticket - Ticket specific options, requires use of the Ticket model
// user   - User specific options, requires use of the User model
$report->filtering = 'ticket';

// Setting out the initial query for the report data before any filtering is applied
$report->query = Ticket::select([DB::raw('ticket.resolved_time - ticket.created_at as resolution_time')])
    ->where('ticket.status_id', Config::get('settings.default_resolved_status'))
    ->groupBy('ticket.id');

/**
 * Returns the data in a chart friendly format - use this to fetch the data and
 * convert it to a format that works with the report charts and tables.
 *
 * @param  Object $report The report data (above data with any custom additional data)
 * @param  mixed  $query  The query after filtering has been applied
 * @return  string           Json string of data
 */
function getData($report, $query)
{
    // Set the start and end time
    if ($report->startDate > 0) {
        $sql = $query->where('ticket.created_at', '>=', $report->startDate);
    }
    if ($report->endDate > 0) {
        $sql = $query->where('ticket.created_at', '<=', $report->endDate);
    }

    // Execute the query
    $result = $query->get()->toArray();

    // Start building returning data array
    $data = [];

    // Add the column headers
    $data[] = [Lang::get('general.time'), Lang::get('report.ticket_count')];

    // To build up the data in predefined periods
    $days = [0, 1, 2, 4, 7];
    $dayCounts = [
        1 => 0,
        2 => 0,
        4 => 0,
        7 => 0,
    ];

    // The count for anything above the cutoff of 7
    $cutoffCount = 0;

    // Add each row of data, ensure it matches the column header
    if (is_array($result)) {
        foreach ($result as $value) {
            $numDays = $value['resolution_time'] / 60 / 60 / 24;

            // Use this to store when we've found what group it fits in
            $found = false;

            foreach ($days as $key => $day) {
                // Is it within this period and the next period start
                if ($numDays >= $day && isset($days[$key + 1]) && ($numDays < $days[$key + 1])) {
                    // Add to the count of this group and declare it found
                    $dayCounts[$days[$key + 1]]++;
                    $found = true;
                    // Don't need to run this foreach any more
                    break;
                }
                // Not in this group, carry on
            }

            // Not found in the groups above, add it to 7+
            if (! $found) {
                $cutoffCount++;
            }
        }
    }

    // Add the rows of data
    $data[] = [Lang::get('report.lt_1_day'), $dayCounts[1]];
    $data[] = [Lang::get('report.1-2_day'), $dayCounts[2]];
    $data[] = [Lang::get('report.2-4_day'), $dayCounts[4]];
    $data[] = [Lang::get('report.4-7_day'), $dayCounts[7]];
    $data[] = [Lang::get('report.7_plus_day'), $cutoffCount];

    // Reset array keys and encode it to JSON
    return json_encode(array_values($data));
}
