<?php
/**
 * User Organisation Custom Field Value Table Migration
 *
 * @package    Database\Migrations\v2_1_0\User
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.1.0
 */
namespace Database\Migrations\v2_1_0\User;

use DB;
use Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Migrations\Migration;

/**
 * Class CreateUserOrganisationCustomfieldValueTable
 *
 * @package    Database\Migrations\v2_1_0\User
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.1.0
 */
class CreateUserOrganisationCustomfieldValueTable extends Migration
{
    /**
     * Mapping of user_organisation column name vs custom field name.
     *
     * @var array
     */
    private $mapping = [
        'website'   => 'Website URL',
        'address'   => 'Address',
        'city'      => 'City',
        'state'     => 'State',
        'postcode'  => 'Postcode',
        'phone'     => 'Phone Number',
    ];

    /**
     * Custom Field data.
     *
     * @var array
     */
    private $fields = [];

    /**
     * CreateUserOrganisationCustomfieldValueTable constructor.
     */
    public function __construct()
    {
        $this->fields = [
            [
                'name'  => $this->mapping['website'],
                'type'  => 8,
                'order' => 1,
            ],
            [
                'name'  => $this->mapping['address'],
                'type'  => 8,
                'order' => 2,
            ],
            [
                'name'  => $this->mapping['city'],
                'type'  => 8,
                'order' => 3,
            ],
            [
                'name'  => $this->mapping['state'],
                'type'  => 8,
                'order' => 4,
            ],
            [
                'name'  => $this->mapping['postcode'],
                'type'  => 8,
                'order' => 5,
            ],
            [
                'name'  => $this->mapping['phone'],
                'type'  => 8,
                'order' => 6,
            ]
        ];
    }

    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::create('user_organisation_customfield_value', function (Blueprint $table) {
            $table->engine = 'InnoDB';
            $table->charset = 'utf8mb4';
            $table->collation = 'utf8mb4_unicode_ci';
            $table->increments('id')->unsigned();
            $table->integer('field_id')->unsigned();
            $table->foreign('field_id')->references('id')->on('user_organisation_customfield')->onDelete('cascade');

            $table->integer('organisation_id')->unsigned();
            $table->foreign('organisation_id')->references('id')->on('user_organisation')->onDelete('cascade');

            $table->text('value')->nullable();
            $table->integer('created_at');
            $table->integer('updated_at')->default(0);
        });

        // Convert user_organisation columns to custom fields.
        $this->migrateColumnsToCustomFields();

        // Drop the existing table data.
        Schema::table('user_organisation', function (Blueprint $table) {
            $table->dropColumn('address');
            $table->dropColumn('city');
            $table->dropColumn('state');
            $table->dropColumn('postcode');
            $table->dropColumn('phone');
            $table->dropColumn('website');
        });
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');

        // Add the user_organisation columns back.
        Schema::table('user_organisation', function (Blueprint $table) {
            $table->string('address')->nullable();
            $table->string('city')->nullable();
            $table->string('state')->nullable();
            $table->string('postcode')->nullable();
            $table->string('phone')->nullable();
            $table->string('website')->nullable();
        });

        // Do the opposite of seed().
        $this->migrateCustomFieldsToColumns();

        // Drop the custom field values.
        Schema::drop('user_organisation_customfield_value');

        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
    }

    /**
     * Convert organisation table columns to custom fields.
     *
     * @return void
     */
    private function migrateColumnsToCustomFields()
    {
        $time = time();

        // Add default user organisation custom fields
        $field1 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[0], [
            'created_at' => $time,
            'updated_at' => $time
        ]));
        $field2 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[1], [
            'created_at' => $time,
            'updated_at' => $time
        ]));
        $field3 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[2], [
            'created_at' => $time,
            'updated_at' => $time
        ]));
        $field4 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[3], [
            'created_at' => $time,
            'updated_at' => $time
        ]));
        $field5 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[4], [
            'created_at' => $time,
            'updated_at' => $time
        ]));
        $field6 = DB::table('user_organisation_customfield')->insertGetId(array_merge($this->fields[5], [
            'created_at' => $time,
            'updated_at' => $time
        ]));

        // Insert values for existing organisations
        $organisations = DB::table('user_organisation')->get();

        // Build insert array.
        $customFieldValues = [];
        foreach ($organisations as $organisation) {
            if (! empty($organisation->website)) {
                $customFieldValues[] = [
                    'field_id'        => $field1,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->website,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
            if (! empty($organisation->address)) {
                $customFieldValues[] = [
                    'field_id'        => $field2,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->address,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
            if (! empty($organisation->city)) {
                $customFieldValues[] = [
                    'field_id'        => $field3,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->city,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
            if (! empty($organisation->state)) {
                $customFieldValues[] = [
                    'field_id'        => $field4,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->state,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
            if (! empty($organisation->postcode)) {
                $customFieldValues[] = [
                    'field_id'        => $field5,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->postcode,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
            if (! empty($organisation->phone)) {
                $customFieldValues[] = [
                    'field_id'        => $field6,
                    'organisation_id' => $organisation->id,
                    'value'           => $organisation->phone,
                    'created_at'      => $time,
                    'updated_at'      => $time
                ];
            }
        }

        // Insert into the database.
        $chunks = array_chunk($customFieldValues, 200);
        foreach ($chunks as $chunk) {
            DB::table('user_organisation_customfield_value')->insert($chunk);
        }
    }

    /**
     * Convert custom fields back to user_organisation column.
     *
     * @return void
     */
    private function migrateCustomFieldsToColumns()
    {
        $customFields = DB::table('user_organisation_customfield')
            ->whereIn('name', array_column($this->fields, 'name'))
            ->pluck('id', 'name')
            ->all();

        // Grab all the custom field values.
        $customFieldValues = DB::table('user_organisation_customfield_value')
            ->whereIn('field_id', array_keys($customFields))
            ->get();

        // Build up a collection of user_organisation records.
        $records = [];
        foreach ($customFieldValues as $customField) {
            // Find the original user_organisation column name.
            $column_name = array_search($customFields[$customField->field_id], $this->mapping);

            if ($column_name !== false) {
                $records[$customField->organisation_id] = array_merge($records[$customField->organisation_id], [
                    $column_name => $customField->value
                ]);
            }
        }

        // Update the organisations.
        foreach ($records as $id => $record) {
            DB::table('user_organisation')->where('id', $id)->update($record);
        }

        // Delete the customfields that we added in migrateColumnsToCustomFields()
        DB::table('user_organisation_customfield')
            ->whereIn('name', array_column($this->fields, 'name'))
            ->delete();
    }
}
