<?php
/**
 * File 2017_05_05_189205_update_email_template_table.php
 *
 * @package    Database\Migrations\v2_2_0\Core
 * @copyright  Copyright (c) 2015-2017 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.2.0
 */
namespace Database\Migrations\v2_2_0\Core;

use App\Modules\Core\Models\EmailTemplate;
use App\Modules\Ticket\Models\Department;
use DB;
use Illuminate\Database\Migrations\Migration;

/**
 * Class UpdateEmailTemplateTable
 *
 * @package    Database\Migrations\v2_2_0\Core
 * @copyright  Copyright (c) 2015-2017 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.2.0
 */
class UpdateEmailTemplateTable extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        // Get custom email templates.
        $templates = EmailTemplate::with('data')
            ->where('type', EmailTemplate::CUSTOM)
            ->get();
        
        // Get the current highest ID in the table.
        $maxId = DB::table('email_template')->max('id');
        if ($maxId < 100) {
            $maxId = 100;
        }
        
        // Any custom email_template with ID > 27 should be moved to ID 100+
        foreach ($templates as $template) {
            if ($template->id >= 100) {
                continue;
            }
            
            // If the ID exists, find one that doesn't.
            while ($this->exists($maxId)) {
                $maxId = $maxId + 1;
            }
            
            // Force email_template ID to 100+.
            DB::table('email_template')->where('id', $template->id)->update([ 'id' => $maxId ]);
                
            // Also update foreign keys.
            DB::table('email_template_data')->where('template_id', $template->id)
                ->update([ 'template_id' => $maxId ]);
            
            DB::table('feedback_form')->where('email_template_id', $template->id)
                ->update([ 'email_template_id' => $maxId ]);
            
            $this->setEscalationRules($template, $maxId);

            $this->setTicketFollowup($template, $maxId);

            $this->setTicketMacro($template, $maxId);
            
            $this->setDepartmentTemplates($template, $maxId);

            DB::table('activity_log')->where('section', 'LIKE', 'core.email_template')
                ->where('rel_id', $template->id)
                ->update([ 'rel_id' => $maxId ]);

            // Increment highest ID.
            $maxId = $maxId + 1;
        }

        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        //
    }

    /**
     * Set new email template ID in the escalation rules.
     *
     * @param EmailTemplate $template
     * @param int           $newId
     * @return void
     */
    private function setEscalationRules(EmailTemplate $template, $newId)
    {
        DB::table('sla_escalation_rule')
            ->whereIn('action', [ 5, 6 ])
            ->where('value_id', $template->id)
            ->update([ 'value_id' => $newId ]);

        DB::table('sla_escalation_rule')
            ->whereIn('action', [ 10, 11 ])
            ->where('value_text', $template->id)
            ->update([ 'value_text' => $newId ]);
    }

    /**
     * Set new email template ID in the ticket follow actions.
     *
     * @param EmailTemplate $template
     * @param int           $newId
     * @return void
     */
    private function setTicketFollowup(EmailTemplate $template, $newId)
    {
        DB::table('ticket_followup_action')
            ->whereIn('action', [ 5, 6 ])
            ->where('value_id', $template->id)
            ->update([ 'value_id' => $newId ]);

        DB::table('ticket_followup_action')
            ->whereIn('action', [ 10, 11 ])
            ->where('value_text', $template->id)
            ->update([ 'value_text' => $newId ]);
    }

    /**
     * Set new email template ID in the ticket macro actions.
     *
     * @param EmailTemplate $template
     * @param int           $newId
     * @return void
     */
    private function setTicketMacro(EmailTemplate $template, $newId)
    {
        DB::table('ticket_macro_action')
            ->whereIn('action', [ 5, 6, 10, 11 ])
            ->where('value_id', $template->id)
            ->update([ 'value_id' => $newId ]);

        DB::table('ticket_macro_action')
            ->whereIn('action', [ 10, 11 ])
            ->where('value_text', $template->id)
            ->update([ 'value_text' => $newId ]);
    }

    /**
     * Set new email template ID in the department email templates.
     *
     * @param EmailTemplate $template
     * @param int           $newId
     * @return void
     */
    private function setDepartmentTemplates(EmailTemplate $template, $newId)
    {
        $departments = Department::whereNull('parent_id')->whereNotNull('email_templates')->get();
        foreach ($departments as $department) {
            $templates = $department->email_templates;

            foreach ($templates as $key => $id) {
                if ($id == $template->id) {
                    $templates[$key] = $newId;
                }
            }

            $department->email_templates = $templates;
            $department->save();
        }
    }

    /**
     * Determine whether an email template already exists.
     *
     * @param  int $id
     * @return bool
     */
    private function exists($id)
    {
        return (bool) DB::table('email_template')->where('id', $id)->count();
    }
}
