<?php
/**
 * File 2018_07_09_122600_update_ticket_macro_table.php
 *
 * @package    Database\Migrations\v2_4_0\Ticket
 * @copyright  Copyright (c) 2015-2018 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.4.0
 */
namespace Database\Migrations\v2_4_0\Ticket;

use App\Modules\Ticket\Events\TicketEvent;
use App\Modules\Ticket\Models\Macro;
use DB;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Database\Migrations\Migration;
use Schema;

/**
 * Class UpdateTicketMacroTable
 *
 * @package    Database\Migrations\v2_4_0\Ticket
 * @copyright  Copyright (c) 2015-2018 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.4.0
 */
class UpdateTicketMacroTable extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        Schema::table('ticket_macro', function (Blueprint $table) {
            $table->tinyInteger('enabled')->default(1)->after('description');
            $table->renameColumn('run_type', 'type');
            $table->tinyInteger('run_at_most')->nullable()->after('all_groups');
        });

        // Create copies of old automatic - new ticket only macros so they are available as manual macros too.
        $macros = Macro::where('type', Macro::TYPE_HOOK)->get();
        foreach ($macros as $macro) {
            $this->createCopy($macro);
        }

        // Create copies of old automatic - scheduled task macros so they are available as manual and hook macros too.
        $macros = Macro::where('type', Macro::TYPE_AUTOMATIC)->get();
        foreach ($macros as $macro) {
            $this->createCopy($macro);
            $this->createCopy($macro, Macro::TYPE_HOOK);
        }

        // For backwards compatibility, set existing automatic macros to run at most one time.
        DB::table('ticket_macro')->where('type', Macro::TYPE_AUTOMATIC)->update([ 'run_at_most' => 1 ]);
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::table('ticket_macro', function (Blueprint $table) {
            $table->dropColumn([ 'enabled', 'run_at_most' ]);
            $table->renameColumn('type', 'run_type');
        });
    }

    /**
     * Create a copy of a macro
     *
     * @param Macro $macro
     * @param int $type
     * @return void
     */
    private function createCopy(Macro $macro, $type = Macro::TYPE_MANUAL)
    {
        $copy = $macro->replicate();
        $copy->type = $type;
        $copy->save();

        $copy->groups()->sync($macro->groups->pluck('id')->toArray());

        foreach ($macro->conditiongroups as $conditionGroup) {
            $newConditionGroup = $conditionGroup->replicate();
            $newConditionGroup->macro_id = $copy->id;
            $newConditionGroup->save();

            foreach ($conditionGroup->conditions as $condition) {
                $newCondition = $condition->replicate();
                $newCondition->group_id = $newConditionGroup->id;
                $newCondition->save();
            }
        }

        foreach ($macro->actions as $action) {
            $newAction = $action->replicate();
            $newAction->macro_id = $copy->id;
            $newAction->save();
        }

        // Have to do this differently as replicate doesn't work on translations
        foreach ($macro->translations as $translation) {
            DB::table('ticket_macro_translation')->insert([
                'macro_id'    => $copy->id,
                'name'        => $translation->name,
                'description' => $translation->description,
                'locale'      => $translation->locale,
            ]);
        }

        if ($type == Macro::TYPE_HOOK) {
            $copy->events()->create([ 'event_id' => TicketEvent::NEW_TICKET ]);
        }
    }
}
