<?php
/**
 * File 2016_12_01_113999_seed_database_table.php
 *
 * @package    Database\Seeds\v2_1_0
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.1.0
 */
namespace Database\Seeds\v2_1_0;

use App;
use DB;
use App\Modules\Core\Controllers\Database\Seed\Seeder;

/**
 * Class SeedDatabaseTable
 *
 * @package    Database\Seeds\v2_1_0
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.1.0
 */
class SeedDatabaseTable extends Seeder
{
    /**
     * List of columns that we expect to be UTF8 not UTF8mb4.
     *
     * @var array
     */
    public static $expected_utf8 = [
        'upload.hash', 'article_attachment.upload_hash', 'ticket_attachment.upload_hash', 'setting.name',
        'plugin.name', 'license.name', 'license.value', 'widget.id', 'widget.name', 'widget_role.widget_id',
        'operator_widget.widget_id', 'user.email', 'mass_email_queue.user_email', 'mass_email_addresses.email'

    ];

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $db = DB::getDatabaseName();

        // Convert to database.
        DB::unprepared("ALTER DATABASE `$db` CHARACTER SET = utf8mb4 COLLATE = utf8mb4_unicode_ci");

        // Column collation check
        $collation_columns = DB::getNonUtf8mb4CollationColumns();
        if (count($collation_columns) !== count(self::$expected_utf8)) {
            $this->note(
                "[ERROR]: Failed to change collation to 'utf8mb4_unicode_ci' on the the following table columns: "
                . implode(', ', self::notUtf8mb4($collation_columns))
                . ". Please contact us for further assistance before proceeding."
            );
        }

        // Table collation check
        if (count($collation_tables = DB::getNonUtf8mb4CollationTables()) !== 0) {
            $this->note(
                "[ERROR]: Failed to change collation to 'utf8mb4_unicode_ci' on the the following tables: "
                . DB::schemaToString($collation_tables)
                . ". Please contact us for further assistance before proceeding."
            );
        }

        // Column charset check
        $charset_columns = DB::getNonUtf8mb4CharsetColumns();
        if (count($charset_columns) !== count(self::$expected_utf8)) {
            $this->note(
                "[ERROR]: Failed to change charset to 'utf8mb4' on the the following table columns: "
                . implode(', ', self::notUtf8mb4($charset_columns))
                . ". Please contact us for further assistance before proceeding."
            );
        }

        // Change config/env/database.php to use new charset etc
        $path = config_path(App::environment().'/database.php');
        if ($this->updateDatabaseConfig($path) === false) {
            $this->note("[ERROR]: Failed to change charset to 'utf8mb4' and collation to 'utf8mb4_unicode_ci'"
                ." in database config file '$path', please change manually.");
        }
    }

    /**
     * Diff the list of provided columns against the expected utf8mb4 columns.
     *
     * @param  array $columns
     * @return array
     */
    public static function notUtf8mb4(array $columns)
    {
        return array_diff(explode(', ', DB::schemaToString($columns)), self::$expected_utf8);
    }

    /**
     * Update the charset and collation in the database config.
     *
     * @param  string $path
     * @return bool
     */
    private function updateDatabaseConfig($path)
    {
        if (! file_exists($path) || ! is_writable($path)) {
            return false;
        }

        $contents = file_get_contents($path);

        // Change charset to utf8mb4
        $contents = preg_replace("/'charset'\s*=>\s*'(.*?)',/", "'charset' => 'utf8mb4',", $contents, 1);

        // Change collation to utf8m4_unicode_ci
        $contents = preg_replace("/'collation'\s*=>\s*'(.*?)',/", "'collation' => 'utf8mb4_unicode_ci',", $contents, 1);

        return (bool) file_put_contents($path, $contents);
    }
}
