<?php
/**
 * File 2020_03_04_172300_seed_plugin_table.php
 *
 * @package    Database\Seeds\v3_0_0
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 3.0.0
 */
namespace Database\Seeds\v3_0_0;

use App\Exceptions\Handler;
use App\Modules\Core\Controllers\Addons\Manager;
use App\Modules\Core\Controllers\Database\Seed\Seeder;
use App\Modules\Core\Models\Repository\SettingsRepository;
use DB;
use Exception;
use PluginFactory;

/**
 * Class SeedPluginTable
 *
 * @package    Database\Seeds\v3_0_0
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 3.0.0
 */
class SeedPluginTable extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // We only want to run this as part of upgrades. The same functionality exists in the install:operator
        // command for new installations.
        if ($this->container->make(SettingsRepository::class)->isInstalled() !== false) {
            // Non-BC changes have been made to plugins and will result in errors post upgrade.
            $this->deactivateCustomPlugins();
        }

        $this->setPluginVersions();
    }

    /**
     * Set plugin versions.
     *
     * @return void
     */
    private function setPluginVersions()
    {
        // We use discover() here because the PluginFactory isn't available for new installs.
        $driver = app()->make(Manager::class)->driver('plugin');
        $plugins = $driver->discover();

        // Set the version for each plugin
        foreach ($plugins as $name => $plugin) {
            if (isset($plugin->getConfig()->version)) {
                DB::table('plugin')->where('name', $name)->update(['version' => $plugin->getConfig()->version]);
            }
        }
    }

    /**
     * Deactivate custom plugins (those not included in core).
     *
     * @return void
     */
    private function deactivateCustomPlugins()
    {
        $plugins = DB::table('plugin')
            ->where('enabled', 1)
            ->get();
        foreach ($plugins as $row) {
            $plugin = PluginFactory::getPlugin($row->name);

            try {
                // Reactivate the plugin (to force the upgrade).
                PluginFactory::deactivate($plugin, true);
            } catch (Exception $e) {
                $this->note("[ERROR]: Failed to deactivate '{$row->name}' plugin. Reason: '{$e->getMessage()}'.");
                Handler::reportException($e);
            }
        }

        if (count($plugins) > 0) {
            $this->note("[NOTICE] The following plugins have been deactivated due to potential compatibility issues"
                . " with this release: " . collect($plugins)->implode('name', ', ') . ". Please see"
                . " https://docs.supportpal.com/current/Upgrading+to+3.0+from+2.6#DevelopmentChanges for more information.");
        }
    }
}
