<?php
/**
 * File 2020_07_07_144718_seed_email_template_data_table.php
 *
 * @copyright  Copyright (c) 2015-2020 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace Database\Seeds\v3_3_0;

use App\Modules\Core\Controllers\Database\Seed\Seeder;
use App\Modules\Core\Models\EmailTemplate;
use App\Modules\Core\Models\Repository\SettingsRepository;
use App\Modules\Ticket\Models\Department;
use Illuminate\Support\Facades\DB;

/**
 * Class SeedEmailTemplateDataTable
 */
class SeedEmailTemplateDataTable extends Seeder
{
    /**
     * Prefix to be appended to email replies.
     *
     * @var string
     */
    protected $replyPrefix = "RE:";

    /**
     * Default ticket notification email subject.
     *
     * @var string
     */
    protected $defaultTicketSubject = '(#{{ ticket.number }}) {{ ticket.subject }}';

    /**
     * Templates pertaining to new ticket notifications i.e the user
     * may not have seen this ticket before hence it cannot be a RE: (reply).
     *
     * @var int[]
     */
    protected $newTicketTemplates = [
        EmailTemplate::USER_TICKET_OPENED_NOTIFICATION,
        EmailTemplate::OPERATOR_TICKET_OPENED_NOTIFICATION,
        EmailTemplate::OPERATOR_NEW_INTERNAL_NOTIFICATION,
        EmailTemplate::OPERATOR_ASSIGNED_NOTIFICATION,
        EmailTemplate::OPERATOR_DEPT_CHANGED_NOTIFICATION,
    ];

    /**
     * Email channel template IDs.
     *
     * @var int[]
     */
    protected $emailChannelTemplates = [
        EmailTemplate::USER_ATTACHMENT_REJECTED_NOTIFICATION,
        EmailTemplate::USER_REGISTERED_ONLY,
        EmailTemplate::USER_TICKET_REPLY_DISABLED_NOTIFICATION,
    ];

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $ids = [1, 4, 18, 19];
        foreach ($ids as $id) {
            $this->updateTemplate(__DIR__ . '/../2.0.0-beta1/emails/template_' . $id . '.twig', $id);
        }

        // New installations only.
        if ($this->container->make(SettingsRepository::class)->isInstalled() === false) {
            // Order of these method calls matters.
            // Subsequent method calls will overwrite the first in some cases.
            $this->updateTicketNotifications();

            $this->updateNewTicketNotifications();
            $this->updateEmailChannelNotifications();
        }
    }

    /**
     * Overwrite email template if the updated time is the same as the created time.
     *
     * @param  string $template
     * @param  int    $id
     * @return void
     */
    private function updateTemplate($template, $id)
    {
        // Check template file exists.
        if (! file_exists($template)) {
            $this->note("[ERROR]: Failed to update email template '$id', template data file could not be found.");
            return;
        }

        // Check if the template has been modified.
        $unmodified = DB::table('email_template_data')
            ->where('template_id', $id)
            ->whereColumn('created_at', 'updated_at')
            ->whereNull('brand_id')
            ->whereNull('language_code')
            ->count();

        if ($unmodified) {
            DB::table('email_template_data')
                ->where('template_id', $id)
                ->whereColumn('created_at', 'updated_at')
                ->whereNull('brand_id')
                ->whereNull('language_code')
                ->update([ 'contents' => file_get_contents($template) ]);
        }
    }

    /**
     * Update email templates for replies to an existing ticket.
     *
     * @return void
     */
    protected function updateTicketNotifications(): void
    {
        $templates = array_merge(Department::getUserTemplates(), Department::getOperatorTemplates());

        DB::table('email_template_data')
            ->whereIn('template_id', $templates)
            ->update(['subject' => "{$this->replyPrefix} {$this->defaultTicketSubject}"]);
    }

    /**
     * Update miscellaneous ticket email templates.
     *
     * @return void
     */
    protected function updateNewTicketNotifications(): void
    {
        DB::table('email_template_data')
            ->whereIn('template_id', $this->newTicketTemplates)
            ->update(['subject' => $this->defaultTicketSubject]);

    }

    /**
     * Update email channel notifications which are a direct reply to
     * the users incoming email.
     *
     * @return void
     */
    protected function updateEmailChannelNotifications(): void
    {
        DB::table('email_template_data')
            ->whereIn('template_id', $this->emailChannelTemplates)
            ->update(['subject' => "{$this->replyPrefix} {{ oSubject }}"]);
    }
}
