<?php declare(strict_types=1);

namespace Database\Seeds\v4_0_0;

use App\Exceptions\Handler;
use App\Modules\Core\Controllers\Addons\Manager\AddonManager;
use App\Modules\Core\Controllers\Database\Seed\Seeder;
use App\Modules\Core\Controllers\Plugins\Facades\PluginFactory;
use App\Modules\Core\Controllers\Plugins\Plugin;
use App\SupportPal;
use Exception;
use Illuminate\Contracts\Container\Container;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use SupportPal\Marketplace\Http\Api\Addons;
use Symfony\Component\Finder\Exception\DirectoryNotFoundException;
use Symfony\Component\Finder\Finder;

use function app_path;
use function base_path;
use function in_array;
use function sprintf;

use const DIRECTORY_SEPARATOR;

class SeedPluginTable extends Seeder
{
    private Addons $addonsApi;

    private AddonManager $addonManager;

    public function __construct(Container $container, Addons $addonsApi, AddonManager $addonManager)
    {
        parent::__construct($container);

        $this->addonsApi = $addonsApi;
        $this->addonManager = $addonManager;
    }

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        try {
            $plugins = Finder::create()
                ->directories()
                ->in(app_path('Plugins'))
                ->depth(0);
        } catch (DirectoryNotFoundException $e) {
            return;
        }

        $toActivate = [];

        foreach ($plugins as $plugin) {
            if ($plugin->getRealPath() === false) {
                continue;
            }

            // Install former core plugins from marketplace to new addons folder.
            $corePlugins = ['BlestaInformation', 'GoogleTranslate', 'SlackNotifications', 'TimeTracking', 'WhmcsInformation', 'WooCommerce'];
            if (in_array($plugin->getBasename(), $corePlugins)) {
                if (! DB::table('plugin')->where('name', $plugin->getBasename())->where('enabled', 1)->count()) {
                    continue;
                }

                $addons = $this->addonsApi->get(['categories' => ['plugins' => [$plugin->getBasename()]]]);

                if ($addons->isEmpty()) {
                    $this->note(sprintf('Unable to find plugin %s at marketplace.', $plugin->getBasename()));

                    continue;
                }

                try {
                    $addon = $addons->first();
                    $file = $this->addonManager->downloadAddon($addon);
                    $this->addonManager->extractAddon($addon, $file);
                    $toActivate[] = $addon;
                } catch (Exception $e) {
                    if ($this->container->runningUnitTests()) {
                        throw $e;
                    }

                    Handler::reportException($e);
                    $this->note(sprintf('Failed to migrate plugin %s to new add-ons framework.', $plugin->getBasename()));
                }

                continue;
            }

            // Migrate other plugins to new addons folder.
            File::moveDirectory(
                $plugin->getRealPath(),
                base_path('addons' . DIRECTORY_SEPARATOR . 'Plugins' . DIRECTORY_SEPARATOR . $plugin->getBasename())
            );
        }

        // Run activate on plugins downloaded from marketplace.
        foreach ($toActivate as $addon) {
            $this->addonManager->activate($addon);
        }
    }
}
