<?php declare(strict_types=1);

namespace Database\Seeds\v4_0_0;

use App\Exceptions\Handler;
use App\Modules\Core\Controllers\Addons\Manager\AddonManager;
use App\Modules\Core\Controllers\Database\Seed\Seeder;
use App\Modules\Ticket\Controllers\Channels\Facades\ChannelFactory;
use App\SupportPal;
use Exception;
use Illuminate\Contracts\Container\Container;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use SupportPal\Marketplace\Http\Api\Addons;
use Symfony\Component\Finder\Exception\DirectoryNotFoundException;
use Symfony\Component\Finder\Finder;

use function app_path;
use function base_path;
use function implode;
use function in_array;
use function sprintf;

use const DIRECTORY_SEPARATOR;

class SeedChannelTable extends Seeder
{
    private Addons $addonsApi;

    private AddonManager $addonManager;

    public function __construct(Container $container, Addons $addonsApi, AddonManager $addonManager)
    {
        parent::__construct($container);

        $this->addonsApi = $addonsApi;
        $this->addonManager = $addonManager;
    }

    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        try {
            $channels = Finder::create()
                ->directories()
                ->in(app_path(implode(DIRECTORY_SEPARATOR, ['Modules', 'Ticket', 'Channels'])))
                ->depth(0)
                ->notName(['API', 'Email', 'Web']);
        } catch (DirectoryNotFoundException $e) {
            $channels = [];
        }

        $toActivate = [];

        // Migrate channels to new addons folder.
        foreach ($channels as $channel) {
            if ($channel->getRealPath() === false) {
                continue;
            }

            // Install former core channels from marketplace to new addons folder.
            $coreChannels = ['Facebook', 'HelpWidget', 'Twitter'];
            if (in_array($channel->getBasename(), $coreChannels)) {
                if (! DB::table('ticket_channel')->where('name', $channel->getBasename())->where('enabled', 1)->count()) {
                    continue;
                }

                $addons = $this->addonsApi->get(['categories' => ['channels' => [$channel->getBasename()]]]);

                if ($addons->isEmpty()) {
                    $this->note(sprintf('Unable to find channel %s at marketplace.', $channel->getBasename()));

                    continue;
                }

                try {
                    $addon = $addons->first();
                    $file = $this->addonManager->downloadAddon($addon);
                    $this->addonManager->extractAddon($addon, $file);
                    $toActivate[] = $addon;
                } catch (Exception $e) {
                    if ($this->container->runningUnitTests()) {
                        throw $e;
                    }

                    Handler::reportException($e);
                    $this->note(sprintf('Failed to migrate channel %s to new add-ons framework.', $channel->getBasename()));
                }

                continue;
            }

            // Migrate other channels to new addons folder.
            File::moveDirectory(
                $channel->getRealPath(),
                base_path('addons' . DIRECTORY_SEPARATOR . 'Channels' . DIRECTORY_SEPARATOR . $channel->getBasename())
            );
        }

        // Run activate on channels downloaded from marketplace.
        foreach ($toActivate as $addon) {
            $this->addonManager->activate($addon);
        }

        DB::table('ticket_channel')->whereIn('name', ['Web', 'Email', 'API', 'HelpWidget'])->update(['show_on_frontend' => 1]);
    }
}
