(function () {
  'use strict';

  var ch = '@';
  var selector = 'sp-mention';
  var global = tinymce.util.Tools.resolve('tinymce.PluginManager');
  var operators = {};
  var timeoutId;

  /**
   * @param {string} html
   * @returns {HTMLElement}
   */
  function htmlToElement(html) {
    var template = document.createElement('template');
    html = html.trim(); // Never return a text node of whitespace as the result
    template.innerHTML = html;

    return template.content.firstChild;
  }

  // https://github.com/tinymce/tinymce/blob/310bafec6dbd453be7746cd2ecf5f5c40b18ed1b/modules/sugar/src/main/ts/ephox/sugar/api/view/Scroll.ts#L17
  var getScrollPosition = function (_DOC) {
    var doc = _DOC !== undefined ? _DOC : document;
    var x = doc.body.scrollLeft || doc.documentElement.scrollLeft;
    var y = doc.body.scrollTop || doc.documentElement.scrollTop;

    return SugarPosition(x, y);
  };

  // https://github.com/tinymce/tinymce/blob/310bafec6dbd453be7746cd2ecf5f5c40b18ed1b/modules/sugar/src/main/ts/ephox/sugar/api/view/SugarPosition.ts#L7
  var SugarPosition = function (left, top) {
    return {
      left: left,
      top: top,
      translate: function (x, y) {
        return SugarPosition(left + x, top + y);
      }
    };
  };

  var getElementDocumentPos = function (editor, elementRect) {
    var n = tinymce.DOM.getPos(editor.getContentAreaContainer());
    var r = getScrollPosition(editor.getDoc());
    elementRect.x += n.x - r.left;
    elementRect.y += n.y - r.top;

    return elementRect;
  };

  /**
   * @param {object} operator
   * @returns {HTMLElement}
   */
  var createCard = function (operator) {
    const selector = 'tox-mentions__card-common';
    const html = $('<div>').append(
      $('<div>', {class: selector, style: 'position: absolute; top: -10000px; left: -10000px'}).append(
        $('<div>', {class: 'card'})
          .append($('<img>', {class: 'sp-flex-initial sp-avatar', src: operator.avatar_url}))
          .append($('<div>', {class: 'sp-flex-grow'})
            .append($('<div>', {class: 'sp-user-name'}).text(operator.formatted_name))
            .append($('<div>', {class: 'sp-description'}).text(operator.email))
          )
      )
    ).html();

    $('.' + selector).remove();

    const element = htmlToElement(html);
    element.addEventListener('mouseleave', function (e) {
      $('.' + selector).remove();
    });

    return document.body.appendChild(element);
  }

  /**
   * Move the card relative to the target element.
   *
   * @param {Editor} editor
   * @param {HTMLElement} card
   * @param {HTMLElement} target
   */
  var reposition = function (editor, card, target) {
    var o = getElementDocumentPos(editor, editor.dom.getRect(target)),
      i = tinymce.DOM.getRect(card);

    var r = tinymce.DOM.getViewPort();
    r.w -= 30; r.h -= 30;

    const testPositions = ["bl-tl", "tl-bl", "tl-br", "bl-tr"];
    var pos = tinymce.geom.Rect.findBestRelativePosition(i, o, r, testPositions);
    var rect = tinymce.geom.Rect.relativePosition(i, o, pos || testPositions[0] || 'bl-tl');

    tinymce.DOM.setStyles(card, {position: "absolute", left: rect.x, top: rect.y});
  }

  var setup = function (editor) {
    editor.on('ResolveName', function (e) {
      if (editor.dom.is(e.target, "." + selector)) {
        e.name = 'mention';
      }
    });

    editor.ui.registry.addAutocompleter('mentions', {
      ch: ch,
      minChars: 1,
      columns: 1,
      highlightOn: ['formatted_name', 'email'],
      onAction: function (autocompleteApi, rng, value) {
        let mention = $('<div>')
          .append(
            $('<data>', {class: selector, contenteditable: false, value: value.id})
              .text(ch + value.formatted_name)
          )
          .html();

        // insert in to the editor
        editor.selection.setRng(rng);
        editor.insertContent(mention);

        // hide the autocompleter
        autocompleteApi.hide();
      },
      fetch: function (pattern) {
        return new tinymce.util.Promise(function (resolve) {
          $.get(laroute.route('ticket.operator.department.search', {id: editor.settings.departmentId}), {brand_id: editor.settings.brandId, s: pattern})
            .done(function(res) {
              var results = res.data.map(function (value) {
                return {
                  type: 'cardmenuitem',
                  value: value, // passed to onAction callback
                  label: value.formatted_name + ' <' + value.email + '>',
                  items: [
                    {
                      type: 'cardcontainer',
                      direction: 'horizontal',
                      items: [
                        {
                          type: 'cardimage',
                          classes: ['tox-sp-avatar'],
                          src: value.avatar_url,
                          alt: value.formatted_name,
                        },
                        {
                          type: 'cardcontainer',
                          direction: 'horizontal',
                          items: [
                            {
                              type: 'cardtext',
                              classes: ['sp-flex-initial', 'tox-sp-mr-1'],
                              text: value.formatted_name,
                              name: 'formatted_name'
                            },
                            {
                              type: 'cardtext',
                              classes: ['tox-sp-description'],
                              text: '<' + value.email + '>',
                              name: 'email'
                            },
                          ]
                        }
                      ],
                    }
                  ]
                }
              })

              resolve(results);
            });
        })
      }
    })
  };

  function Plugin () {
    global.add('mentions', function (editor) {
      setup(editor);
      return {};
    });
  }

  Plugin();

}());