<?php declare(strict_types=1);

/**
 * File Addon.php
 *
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Addons;

use Illuminate\Console\Application;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Route;
use InvalidArgumentException;
use RuntimeException;
use SupportPal\Addons\Config as AddonConfig;
use SupportPal\Addons\Models\AddonModel;
use SupportPal\Addons\Models\AddonSetting;
use SupportPal\Core\Http\Controllers\Controller;

use function array_filter;
use function basename;
use function file_exists;
use function implode;
use function in_array;
use function sprintf;
use function strtolower;
use function version_compare;

/**
 * Class Addon
 */
abstract class Addon extends Controller
{
    const INACTIVE = 0;
    const ACTIVE = 1;
    const UPGRADE_AVAILABLE = 2;
    const UNLICENSED = 3;

    /**
     * A list of configuration variables.
     *
     * @var AddonConfig
     */
    protected $config = [];

    /**
     * Unique identifier for the add-on.
     *
     * @var string
     */
    protected $identifier = '';

    /**
     * Absolute path to the base of the add-on.
     *
     * @var string
     */
    protected $path = '';

    /**
     * Settings page route.
     *
     * @var \Illuminate\Routing\Route
     */
    protected $settingsRoute = null;

    private int $status = self::INACTIVE;

    /**
     * Add-ons can run an installation routine when they are activated. This
     * will typically include adding default values, initialising database tables
     * and so on.
     *
     * @return bool
     */
    abstract public function activate();

    /**
     * Deactivating serves as temporarily disabling the add-on, but the files still
     * remain. This function should typically clear any caches and temporary directories.
     *
     * @return bool
     */
    abstract public function deactivate();

    /**
     * When an add-on is uninstalled, it should be completely removed as if it never
     * was there. This function should delete any created database tables, and any files
     * created outside of the add-on directory.
     *
     * @return bool
     */
    abstract public function uninstall();

    public function setStatus(int $status): self
    {
        if (! in_array($status, [self::INACTIVE, self::ACTIVE, self::UPGRADE_AVAILABLE, self::UNLICENSED])) {
            throw new InvalidArgumentException('Invalid status given.');
        }

        $this->status = $status;

        return $this;
    }

    public function getStatus(): int
    {
        return $this->status;
    }

    /**
     * Register that the add-on has a settings page. This function will ensure that
     * the page is added to the operator panel and accessible to the user.
     *
     * @param  string $route
     */
    public function registerSetting($route)
    {
        $this->settingsRoute = $route;
    }

    /**
     * Set add-on configuration options
     *
     * @param  string $file Location of the config file
     * @throws FileNotFoundException
     * @throws Exceptions\BadConfigException
     */
    public function setConfig($file = '')
    {
        if (! file_exists($file)) {
            throw new FileNotFoundException($file);
        }

        $config = include $file;
        $this->config = new AddonConfig($config);
    }

    /**
     * Get a add-on's configuration
     *
     * @return AddonConfig
     */
    public function getConfig()
    {
        return $this->config;
    }

    /**
     * Get the path to the add-on
     *
     * @return string
     */
    public function getPath()
    {
        return $this->path;
    }

    /**
     * Set the absolute path to the base of the add-on
     *
     * @param  string $path
     */
    public function setPath($path)
    {
        $this->path = $path;
        $this->identifier = basename($path);
    }

    /**
     * Set add-on identifier
     *
     * @param  string $identifier
     */
    public function setIdentifier($identifier)
    {
        $this->identifier = $identifier;
    }

    /**
     * Get add-on identifier
     *
     * @return string
     */
    public function getIdentifier()
    {
        return $this->identifier;
    }

    /**
     * Get the setting page route
     *
     * @return string
     */
    public function getSettingsRoute()
    {
        return $this->settingsRoute;
    }

    /**
     * Get the settings from the config.
     *
     * @return array
     */
    public function getSettings()
    {
        return $this->getModel('settings');
    }

    /**
     * Alias of getSettings.
     *
     * @return array
     */
    public function settings()
    {
        return $this->getSettings();
    }

    /**
     * Adds or updates a specific channel setting
     *
     * @param  string $name  Name of setting
     * @param  mixed  $value Value of setting
     * @return AddonSetting
     */
    public function addSetting($name, $value)
    {
        if ($value === null) {
            throw new InvalidArgumentException('The value cannot be null.');
        }

        $model = $this->settingsModel();

        return $model::addSetting($this->getModel('id'), $name, $value);
    }

    /**
     * Removes all settings for channel
     *
     * @return bool|null
     */
    public function removeSettings()
    {
        $model = $this->settingsModel();

        return $model::removeSettings($this->getModel('id'));
    }

    /**
     * Removes a specific setting for channel
     *
     * @param  string $name Name of setting
     * @return bool|null
     */
    public function removeSetting($name)
    {
        $model = $this->settingsModel();

        return $model::removeSetting($this->getModel('id'), $name);
    }

    /**
     * Checks if a setting exists
     *
     * @param  string $name  Name of setting
     * @param  string $value Value of setting, optional
     * @return bool
     */
    public function hasSetting($name, $value = null)
    {
        $model = $this->settingsModel();

        return $model::hasSetting($this->getModel('id'), $name, $value);
    }

    /**
     * Checks if the add-on currently has settings stored
     *
     * @return bool
     */
    public function hasSettings()
    {
        $class = $this->modelClass();
        $settingsClass = $this->settingsModel();

        return (bool) $settingsClass::where($class::foreignKeyName(), $this->getModel('id'))->count();
    }

    /**
     * Is the add-on currently enabled?
     *
     * @return bool
     */
    public function isEnabled(): bool
    {
        $dbConfig = $this->getModel();

        return (bool) Arr::get($dbConfig, 'enabled', false);
    }

    /**
     * Fetch the currently installed version stored in the database (loaded from cache). Will return null if the add-on
     * has never been activated.
     *
     * @return string|null
     */
    public function installedVersion()
    {
        $dbConfig = $this->getModel();

        return Arr::get($dbConfig, 'version');
    }

    /**
     * Check whether an add-on has an update available.
     *
     * @return bool
     */
    public function hasUpgradeAvailable()
    {
        $config = $this->config;
        if (empty($config['version'])) {
            return false;
        }

        // Get database version.
        $existingVersion = $this->installedVersion();
        if (empty($existingVersion)) {
            return false;
        }

        // Database version is lower than the file version (new version).
        return version_compare($existingVersion, $config['version']) === -1;
    }

    /**
     * Get a model config item.
     *
     * @param  string $key
     * @return mixed
     */
    public function getModel($key = null)
    {
        /** @var AddonModel $class */
        $class = $this->modelClass();

        $name = strtolower($this->identifier);

        return model_config($class::getCacheKey(), implode('.', array_filter([$name, $key])));
    }

    /**
     * Register a console command.
     *
     * @param  string $command
     * @return void
     */
    protected function registerCommand($command)
    {
        Application::starting(function ($artisan) use ($command) {
            $artisan->add($artisan->resolve($command));
        });
    }

    /**
     * Add middleware to a route.
     *
     * @param  string $name
     * @param  string $class
     * @return void
     */
    protected function addMiddlewareToRoute($name, $class)
    {
        $route = Route::getRoutes()->getByName($name);
        if ($route === null) {
            throw new RuntimeException(sprintf('Missing route \'%s\'.', $name));
        }

        $route->middleware($class);
    }

    /**
     * Namespace of the associated model.
     *
     * @return AddonModel|string
     */
    abstract protected function modelClass();

    /**
     * Namespace of the associated settings model.
     *
     * @return AddonSetting|string
     */
    protected function settingsModel()
    {
        /** @var AddonModel $class */
        $class = $this->modelClass();

        return $class::settingsModel();
    }
}
