<?php declare(strict_types=1);

namespace SupportPal\Addons;

use ArrayAccess;
use ArrayIterator;
use BadMethodCallException;
use Countable;
use Illuminate\Support\Collection;
use IteratorAggregate;
use SupportPal\Addons\Drivers\Driver;
use Traversable;

use function collect;
use function count;
use function in_array;
use function method_exists;
use function sprintf;

/**
 * @mixin Collection
 */
class AddonCollection implements ArrayAccess, Countable, IteratorAggregate
{
    private Driver $driver;

    /** @var array<Addon> */
    private array $items;

    /**
     * @param array<Addon> $items
     */
    public function __construct(Driver $driver, array $items)
    {
        $this->driver = $driver;
        $this->items = $items;
    }

    /**
     * @return self<Addon>
     */
    public function onlyUpgradeAvailable(): self
    {
        return $this->filter(function (Addon $addon) {
            return $addon->hasUpgradeAvailable();
        });
    }

    /**
     * @return self<Addon>
     */
    public function withoutUpgradeAvailable(): self
    {
        return $this->filter(function (Addon $addon) {
            return ! $addon->hasUpgradeAvailable();
        });
    }

    /**
     * @return self<Addon>
     */
    public function withoutInactive(): self
    {
        $active = $this->driver->getActive();

        return $this->filter(function (Addon $addon) use ($active) {
            return in_array($addon->getIdentifier(), $active, true);
        });
    }

    /**
     * @param string $method
     * @param mixed[] $parameters
     * @return mixed
     */
    public function __call($method, $parameters)
    {
        $collection = collect($this->items);

        if (method_exists($collection, $method)) {
            $value = $collection->$method(...$parameters);

            if ($value instanceof Collection) {
                return new self($this->driver, $value->all());
            }

            $this->items = $collection->all();

            return $value;
        }

        throw new BadMethodCallException(sprintf('Call to undefined method %s::%s()', static::class, $method));
    }

    /**
     * Determine if an item exists at an offset.
     *
     * @param  mixed  $key
     * @return bool
     */
    #[\ReturnTypeWillChange]
    public function offsetExists($key)
    {
        return isset($this->items[$key]);
    }

    /**
     * Get an item at a given offset.
     *
     * @param  mixed  $key
     * @return mixed
     */
    #[\ReturnTypeWillChange]
    public function offsetGet($key)
    {
        return $this->items[$key];
    }

    /**
     * Set the item at a given offset.
     *
     * @param  mixed  $key
     * @param  mixed  $value
     * @return void
     */
    #[\ReturnTypeWillChange]
    public function offsetSet($key, $value)
    {
        if ($key === null) {
            $this->items[] = $value;
        } else {
            $this->items[$key] = $value;
        }
    }

    /**
     * Unset the item at a given offset.
     *
     * @param  string  $key
     * @return void
     */
    #[\ReturnTypeWillChange]
    public function offsetUnset($key)
    {
        unset($this->items[$key]);
    }

    public function count(): int
    {
        return count($this->items);
    }

    public function getIterator(): Traversable
    {
        return new ArrayIterator($this->items);
    }
}
