<?php declare(strict_types=1);

/**
 * File Config.php
 *
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Addons;

use ArrayAccess;
use Illuminate\Support\Facades\Lang;
use SupportPal\Addons\Exceptions\BadConfigException;

use function in_array;
use function is_string;

/**
 * Class Config
 */
class Config implements ArrayAccess
{
    /**
     * Configuration options, indexed by configuration key.
     *
     * @var mixed[]
     */
    private $data = [];

    /**
     * List of translatable fields.
     *
     * @var string[]
     */
    private $translatable = ['name', 'description'];

    /**
     * Initialise the add-on config
     *
     * @param  mixed[] $data
     * @throws BadConfigException
     */
    public function __construct(array $data = [])
    {
        $this->data = $data;

        // Check whether the configuration is valid
        if (! $this->isValid()) {
            throw new BadConfigException('Add-on configuration must contain a non-empty name.');
        }
    }

    /**
     * Check whether the provided configuration is valid
     *
     * @return bool
     */
    public function isValid()
    {
        $name = isset($this->data['name']) ? $this->data['name'] : null;

        return ! empty($name) && is_string($name);
    }

    /**
     * Get the original version of an attribute from the config, bypassing the magic methods.
     *
     * @param  string $key
     * @param  mixed  $default
     * @return mixed
     */
    public function getOriginal($key, $default = null)
    {
        return isset($this->data[$key]) ? $this->data[$key] : $default;
    }

    /**
     * Get a data by key
     *
     * @param  string $key The key data to retrieve
     * @return mixed empty string if the key doesn't exist
     */
    public function __get($key)
    {
        return isset($this->data[$key]) ? $this->getAttribute($key) : '';
    }

    /**
     * Assigns a value to the specified data
     *
     * @param  string $key   The data key to assign the value to
     * @param  mixed  $value The value to set
     */
    public function __set($key, $value)
    {
        $this->data[$key] = $value;
    }

    /**
     * Whether or not an data exists by key
     *
     * @param  string $key An data key to check for
     * @return boolean
     */
    public function __isset($key)
    {
        return isset($this->data[$key]);
    }

    /**
     * Unsets an data by key
     *
     * @param  string $key The key to unset
     */
    public function __unset($key)
    {
        unset($this->data[$key]);
    }

    /**
     * Assigns a value to the specified offset
     *
     * @param  string $offset The offset to assign the value to
     * @param  mixed  $value  The value to set
     */
    public function offsetSet($offset, $value): void
    {
        if ($offset === null) {
            $this->data[] = $value;
        } else {
            $this->data[$offset] = $value;
        }
    }

    /**
     * Whether or not an offset exists
     *
     * @param  string $offset An offset to check for
     * @return boolean
     */
    public function offsetExists($offset): bool
    {
        return isset($this->data[$offset]);
    }

    /**
     * Unsets an offset
     *
     * @param  string $offset The offset to unset
     */
    public function offsetUnset($offset): void
    {
        if (! $this->offsetExists($offset)) {
            return;
        }

        unset($this->data[$offset]);
    }

    /**
     * Returns the value at specified offset
     *
     * @param  string $offset The offset to retrieve
     * @return mixed empty string if the key doesn't exist
     */
    #[\ReturnTypeWillChange]
    public function offsetGet($offset)
    {
        return $this->offsetExists($offset) ? $this->getAttribute($offset) : '';
    }

    /**
     * Get attribute from data array.
     *
     * @param  string $key
     * @return mixed
     */
    private function getAttribute($key)
    {
        if (in_array($key, $this->translatable)) {
            return Lang::get($this->data[$key]);
        }

        return $this->data[$key];
    }
}
