<?php declare(strict_types=1);

/**
 * File Driver.php
 *
 * @copyright  Copyright (c) 2015-2019 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Addons\Drivers;

use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use League\Flysystem\FileNotFoundException;
use SupportPal\Addons\Addon;
use SupportPal\Addons\AddonCollection;
use SupportPal\Addons\Exceptions\BadConfigException;
use SupportPal\Addons\Exceptions\NoClassDefFoundException;
use SupportPal\Addons\Models\AddonModel;
use SupportPal\Addons\Support\Files;

use function array_filter;
use function in_array;
use function sprintf;

/**
 * Class Driver
 */
abstract class Driver
{
    /**
     * Laravel instance.
     *
     * @var Application
     */
    protected $app;

    /**
     * List of add-ons.
     *
     * @var Addon[]
     */
    protected $instances = [];

    /**
     * Driver constructor.
     *
     * @param  Application $app
     */
    public function __construct(Application $app)
    {
        $this->app = $app;
    }

    /**
     * Iterate over the add-on path and return a list of add-ons from the directory
     *
     * @return AddonCollection<Addon>
     */
    abstract public function discover(): AddonCollection;

    /**
     * Initialise and register add-ons in the driver.
     *
     * @param AddonCollection<Addon> $addons
     */
    abstract public function boot(AddonCollection $addons): void;

    /**
     * Synchronise the add-on on the file system with their respective record in the database.
     *
     * @return array<AddonModel>
     * @throws Exception
     */
    abstract public function synchronise(): array;

    /**
     * Initialise an add-on.
     *
     * @param  Addon $addon
     * @return Addon
     * @throws BadConfigException
     * @throws NoClassDefFoundException
     * @throws FileNotFoundException
     */
    abstract protected function initialise(Addon $addon);

    /**
     * Check permissions for the given model.
     *
     * @param  AddonModel $model
     * @return bool
     */
    abstract public function checkPermission(AddonModel $model);

    /**
     * Absolute path to directory where the add-ons are stored.
     *
     * @return string
     */
    abstract public function getDirectory();

    /**
     * Namespace of the associated model.
     *
     * @return string
     */
    abstract public function getModel();

    /**
     * Namespace of the deactivated add-on.
     *
     * @return string
     */
    abstract public function getDeactivatedAddon();

    /**
     * Clear the cache related the add-on.
     */
    abstract public function clearCache(): void;

    /**
     * Register a new add-on.
     *
     * @param  Addon $addon
     * @return void
     */
    public function register(Addon $addon)
    {
        $this->instances[$addon->getIdentifier()] = $addon;
    }

    /**
     * Get a registered add-on
     *
     * @param  string $identifier
     * @return Addon|null Returns the add-on or NULL if it's not registered
     */
    public function getInstance($identifier)
    {
        if (isset($this->instances[$identifier])) {
            return $this->instances[$identifier];
        }

        return null;
    }

    /**
     * List of valid add-ons that exist on the file system
     *
     * @param  bool $filter_active
     * @return Addon[]
     */
    public function allInstances($filter_active = false)
    {
        $instances = $this->instances;

        // Only get active add-ons
        if ($filter_active) {
            $active = $this->getActive();
            $instances = array_filter($instances, function (Addon $p) use ($active) {
                return in_array($p->getIdentifier(), $active);
            });
        }

        return $instances;
    }

    /**
     * Fetch the list of active add-ons from the database
     *
     * @return array
     */
    public function getActive()
    {
        return $this->getModelConfig()
            ->filter(function ($addon) {
                return $addon['enabled'];
            })
            ->pluck('name')
            ->toArray();
    }

    /**
     * Get the model cache config.
     *
     * @return Collection
     */
    protected function getModelConfig()
    {
        /** @var AddonModel $model */
        $model = $this->getModel();
        $callback = function () use ($model) {
            return model_config($model::getCacheKey(), null);
        };

        // Previously add-ons used an array and channels used a collection, this was normalised to all use collections.
        // This code resets the cache otherwise "must be instance of Collection" gets thrown.
        if (($config = $callback()) instanceof Collection) {
            return $config;
        }

        $this->clearCache();

        return $callback();
    }

    /**
     * Activate an add-on.
     *
     * @param  Addon $addon
     * @param  bool  $skipPermissions
     * @return bool
     */
    public function activate(Addon $addon, $skipPermissions = false)
    {
        try {
            /** @var AddonModel $model */
            $model = $this->getModel();

            // Fetch add-on model
            $model = $model::where('name', $addon->getIdentifier())->firstOrFail();

            if (! $skipPermissions && ! $this->checkPermission($model)) {
                // Do not have permission to manage add-on
                return false;
            }

            // Initialise the add-on components (routes, namespaces)
            $addon = $this->initialise($addon);

            // Enable the add-on in the DB
            $model->enabled = 1;
            $model->save();

            // Update the loaded array
            $this->register($addon);

            // Run the activation function on the add-on
            if (! $addon->activate()) {
                throw new Exception('Add-on blocked the operation.');
            }

            // Successfully activated, remove upgrade flag.
            $model->upgrade_available = 0;
            $model->version = $addon->getConfig()->version;
            $model->save();

            // Update the cache.
            $this->clearCache();

            return true;
        } catch (Exception $e) {
            Log::warning(sprintf('[Add-ons] Unable to activate add-on - %s', $e->getMessage()), ['exception' => $e]);

            return false;
        }
    }

    /**
     * Deactivate an add-on.
     *
     * @param  Addon $addon
     * @param  bool  $skipPermissions
     * @return bool
     */
    public function deactivate(Addon $addon, $skipPermissions = false)
    {
        try {
            /** @var AddonModel $model */
            $model = $this->getModel();

            // Check if the add-on is currently active
            $model = $model::where('name', $addon->getIdentifier())->whereEnabled()->first();

            if ($model !== null) {
                if (! $skipPermissions && ! $this->checkPermission($model)) {
                    // Do not have permission to manage add-on
                    return false;
                }

                // Initialise add-on if it's been unloaded due to be unlicensed.
                if ($addon->getStatus() === Addon::UNLICENSED) {
                    $deactivatedAddon = $this->getDeactivatedAddon();

                    if ($addon instanceof $deactivatedAddon) {
                        $addon = $this->initialise($addon);
                    }
                }

                // Disable the add-on in the DB
                $model->enabled = 0;
                if (! $model->save()) {
                    throw new Exception('Failed to save to the database.');
                }

                // Run the activation function on the add-on
                if (! $addon->deactivate()) {
                    throw new Exception('Add-on blocked the operation.');
                }

                // Update the cache.
                $this->clearCache();

                return true;
            }

            // Can't find add-on
            throw new Exception('Model not found or already deactivated.');
        } catch (Exception $e) {
            Log::warning(sprintf('[Add-on] Unable to deactivate add-on. Message: %s.', $e->getMessage()), ['exception' => $e]);

            return false;
        }
    }

    /**
     * Check whether an add-on has an update available.
     *
     * @param  Addon $addon
     * @return bool
     */
    public function hasUpdateAvailable(Addon $addon)
    {
        return $addon->hasUpgradeAvailable();
    }

    /**
     * Add to the add-on directory and call the install function.
     *
     * @param  Addon $addon
     * @return boolean
     */
    public function install(Addon $addon)
    {
        // Download a zip file containing the add-on

        // Extract it to the correct add-ons directory

        // Add a record to correct add-on table
    }

    /**
     * Delete from the add-on directory and call the uninstall function.
     *
     * @param  Addon $addon
     * @return boolean
     */
    public function uninstall(Addon $addon)
    {
        try {
            /** @var AddonModel $model */
            $model = $this->getModel();

            // Check if the add-on is currently active
            $model = $model::where('name', $addon->getIdentifier())->firstOrFail();

            if (! $this->checkPermission($model)) {
                // Do not have permission to manage add-on
                return false;
            }

            // Initialise add-on if it's been unloaded due to be unlicensed.
            if ($addon->getStatus() === Addon::UNLICENSED) {
                $deactivatedAddon = $this->getDeactivatedAddon();

                if ($addon instanceof $deactivatedAddon) {
                    $addon = $this->initialise($addon);
                }
            }

            // Invoke the uninstall function
            $addon->uninstall();

            // Delete the directory
            Files::deleteDir($addon->getPath());

            // Remove the add-on table record
            $model->delete();

            // Flush the cache
            $this->clearCache();

            return true;
        } catch (Exception $e) {
            Log::warning(sprintf('[Add-ons] Unable to uninstall add-on. Message: %s', $e->getMessage()), ['exception' => $e]);

            return false;
        }
    }
}
