<?php declare(strict_types=1);

/**
 * File AddonModel.php
 *
 * @copyright  Copyright (c) 2015-2019 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Addons\Models;

use Closure;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;
use SupportPal\Addons\Addon;
use SupportPal\Core\Database\Models\BaseModel;

use function collect;
use function strtolower;

/**
 * Class AddonModel
 */
abstract class AddonModel extends BaseModel
{
    /**
     * Which fields are fillable
     *
     * @var array
     */
    protected $fillable = ['name', 'enabled', 'upgrade_available', 'version'];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'name'              => 'string',
        'enabled'           => 'int',
        'upgrade_available' => 'int',
        'version'           => 'string',
        'created_at'        => 'int',
        'updated_at'        => 'int',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['formatted_name'];

    /**
     * Scope to fetch only enabled add-ons
     *
     * @param  Builder $query
     * @return mixed
     */
    public function scopeWhereEnabled($query)
    {
        return $query->where($this->getTable() . '.enabled', 1);
    }

    /**
     * Fetch the add-on settings
     *
     * @return HasMany
     */
    public function settings()
    {
        $settingsModel = static::settingsModel();

        if (! empty($settingsModel)) {
            return $this->hasMany($settingsModel, static::foreignKeyName());
        }
    }

    /**
     * Set the name attribute.
     *
     * @param string $name
     * @return void
     */
    public function setNameAttribute(string $name): void
    {
        $this->attributes['name'] = Str::substr($name, 0, 255);
    }

    /**
     * Set the version attribute.
     *
     * @param string|null $version
     * @return void
     */
    public function setVersionAttribute(?string $version): void
    {
        if ($version !== null) {
            $version = Str::substr($version, 0, 255);
        }

        $this->attributes['version'] = $version;
    }

    /**
     * Get the add-on formatted name.
     *
     * @return string|null
     */
    public function getFormattedNameAttribute()
    {
        $addon = $this->getAddon();
        if ($addon === null) {
            return null;
        }

        return $addon->getConfig()['name'];
    }

    /**
     * Get all add-ons with their settings.
     *
     * @return Collection
     */
    public static function toList(): Collection
    {
        $settingsModel = static::settingsModel();

        if (empty($settingsModel)) {
            $addons = self::get();
        } else {
            $addons = self::with('settings')->get();

            // Add each item from settings table
            foreach ($addons as $addon) {
                $addon->setRelation('settings', $addon->flattenSettings());
            }
        }

        return $addons->keyBy(function ($item) {
            return strtolower($item->name);
        });
    }

    /**
     * Flatten the settings collection to an array with just name and value.
     *
     * @return array
     */
    public function flattenSettings()
    {
        return $this->settings->pluck('value', 'name')->all();
    }

    /**
     * Set the add-on cache and config.
     *
     * @param  Closure|null $callback
     * @param  bool         $override
     * @return void
     */
    public static function setCache(?Closure $callback = null, $override = false)
    {
        // If there is no callback, fall back to all channels.
        if ($callback === null) {
            $callback = function () {
                return static::toList();
            };
        }

        try {
            // Update the cache.
            if ($override || ! Cache::has(static::getCacheKey())) {
                Cache::put(static::getCacheKey(), $value = $callback(), 3600);
            } else {
                $value = Cache::get(static::getCacheKey());
            }

            // Update the config.
            Config::set(static::getCacheKey(), $value);
        } catch (Exception $e) {
            Config::set(static::getCacheKey(), collect());
        }
    }

    /**
     * Get the key used to store model data in the cache.
     *
     * @return string
     */
    public static function getCacheKey()
    {
        return static::CACHE_KEY;
    }

    /**
     * Namespace of the settings model.
     *
     * @return string|null
     */
    public static function settingsModel()
    {
        return static::SETTINGS_RELATION;
    }

    /**
     * Name of the column in the settings table that references this model via foreign key.
     *
     * @return string
     */
    public static function foreignKeyName()
    {
        return static::RELATION_COLUMN_NAME;
    }

    /**
     * Add-on instance associated with the model.
     *
     * @return Addon
     */
    abstract public function getAddon();
}
