<?php declare(strict_types=1);

/**
 * File AddonSetting.php
 *
 * @copyright  Copyright (c) 2015-2019 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 */
namespace SupportPal\Addons\Models;

use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use SupportPal\Core\Database\Models\BaseModel;

/**
 * Class AddonSetting
 */
abstract class AddonSetting extends BaseModel
{
    /**
     * AddonSetting constructor.
     *
     * @param  mixed[] $attributes
     */
    public function __construct(array $attributes = [])
    {
        $this->fillable = $this->setFillable();
        $this->casts = $this->setCasts();

        parent::__construct($attributes);
    }

    /**
     * Set the fillable fields for this model.
     *
     * @return array
     */
    protected function setFillable()
    {
        $class = static::modelRelation();

        return [$class::foreignKeyName(), 'name', 'value'];
    }

    /**
     * Set the casts for this model.
     *
     * @return array
     */
    protected function setCasts()
    {
        $class = static::modelRelation();

        return [
            $class::foreignKeyName() => 'int',
            'name'                   => 'string',
            'value'                  => 'string',
            'created_at'             => 'int',
            'updated_at'             => 'int',
        ];
    }

    /**
     * Adds or updates setting in table
     *
     * @param  int    $id    Model ID
     * @param  string $name  Name of setting
     * @param  string $value Value of setting
     * @return static
     */
    public static function addSetting($id, $name, $value)
    {
        $class = static::modelRelation();

        /** @var static $model */
        $model = self::updateOrCreate([
            $class::foreignKeyName() => $id,
            'name'                   => $name
        ], [
            'value' => $value
        ]);

        // Flush cache.
        Cache::forget($class::getCacheKey());

        return $model;
    }

    /**
     * Removes settings for an add-on in table
     *
     * @param  int $id Model ID
     * @return bool|null
     */
    public static function removeSettings($id)
    {
        $class = static::modelRelation();

        // Delete all settings for this channel
        $result = self::where($class::foreignKeyName(), $id)->delete();

        // Flush cache.
        Cache::forget($class::getCacheKey());

        return $result;
    }

    /**
     * Removes a specific setting for an add-on in table
     *
     * @param  int    $id   Model ID
     * @param  string $name Name of setting
     * @return bool|null
     */
    public static function removeSetting($id, $name)
    {
        $class = static::modelRelation();

        // Delete the named setting for the add-on
        $result = self::where($class::foreignKeyName(), $id)
            ->where('name', $name)
            ->delete();

        // Flush cache.
        Cache::forget($class::getCacheKey());

        return $result;
    }

    /**
     * Checks if a setting exists
     *
     * @param  int    $id    Model ID
     * @param  string $name  Name of setting
     * @param  string $value Value of setting, optional
     * @return bool
     */
    public static function hasSetting($id, $name, $value = null)
    {
        $class = static::modelRelation();

        $query = self::where($class::foreignKeyName(), $id)->where('name', $name);

        if (isset($value)) {
            $query = $query->where('value', $value);
        }

        // Check if there is a row for the query
        return $query->count() > 0;
    }

    /**
     * The model that the settings are associated with.
     *
     * @return BelongsTo
     */
    public function relation()
    {
        return $this->belongsTo(static::MODEL_RELATION);
    }

    /**
     * Namespace of the model that the settings are associated with.
     *
     * @return AddonModel|string
     */
    protected static function modelRelation()
    {
        return static::MODEL_RELATION;
    }
}
