<?php declare(strict_types=1);

/**
 * BaseModel Model
 *
 * A modified version of the default model, changed to use UNIX timestamps
 *
 * @package    SupportPal\Core\Database\Models
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.0.0
 */
namespace SupportPal\Core\Database\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use SupportPal\Core\Database\Traits\OrderingHelpers;

use function array_diff;
use function array_keys;
use function array_map;
use function array_merge;
use function collect;
use function implode;
use function in_array;
use function is_array;
use function reset;
use function sprintf;
use function strrpos;
use function substr;
use function trim;
use function vsprintf;
use function with;

/**
 * Class BaseModel
 *
 * @package    SupportPal\Core\Database\Models
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.0.0
 */
class BaseModel extends Model
{
    use OrderingHelpers;

    /**
     * Automatically order all queries using the specified column
     *
     * @var string|bool
     */
    protected $automaticOrdering = false;

    /**
     * Indicates if the model has update and creation timestamps
     *
     * @var bool
     */
    public $timestamps = true;

    /**
     * The storage format of the model's date columns.
     *
     * @var string
     */
    protected $dateFormat = 'U';

    /**
     * Get a new query builder for the model's table.
     *
     * @return Builder
     */
    public function newQuery()
    {
        $query = parent::newQuery();

        if (! empty($this->automaticOrdering) && in_array($this->automaticOrdering, $this->fillable)) {
            $query->orderBy($this->automaticOrdering);
        }

        return $query;
    }

    /**
     * Get the name of the table
     *
     * @return string
     */
    public static function getTableName()
    {
        return with(new static)->getTable();
    }

    /**
     * Don't mutate our UNIX timestamp on fetch
     *
     * @return array
     */
    public function getDates()
    {
        return [];
    }

    /**
     * Set the value of the "created at" attribute.
     *
     * @param  mixed  $value
     * @return $this
     */
    public function setCreatedAt($value)
    {
        if ($value instanceof Carbon) {
            $value = $value->getTimestamp();
        }

        return parent::setCreatedAt($value);
    }

    /**
     * Set the value of the "updated at" attribute.
     *
     * @param  mixed  $value
     * @return $this
     */
    public function setUpdatedAt($value)
    {
        if ($value instanceof Carbon) {
            $value = $value->getTimestamp();
        }

        return parent::setUpdatedAt($value);
    }

    /**
     * Set the original attributes (used in transaction deferred event).
     *
     * @param mixed[] $original
     * @return $this
     */
    public function setOriginal($original)
    {
        $this->original = $original;

        return $this;
    }

    /**
     * Save the model to the database, without firing any events.
     *
     * @param mixed[] $options
     * @return bool
     */
    public function saveWithoutEvents(array $options = [])
    {
        return static::withoutEvents(function () use ($options) {
            return $this->save($options);
        });
    }

    /**
     * Fire the given event for the model.
     *
     * @param  string  $event
     * @param  bool    $halt
     * @return mixed
     */
    public function fireEvent($event, $halt = true)
    {
        return $this->fireModelEvent($event, $halt);
    }

    /**
     * Insert IGNORE MySQL command.
     *
     * @param  mixed[] $values
     * @return bool
     */
    public static function insertIgnore(array $values)
    {
        if (empty($values)) {
            return true;
        }

        [$query, $bindings] = self::compileInsert($values, true);

        return DB::insert($query, $bindings);
    }

    /**
     * Insert command with ON DUPLICATE KEY UPDATE functionality.
     *
     * Accepts to methods of UPDATE:
     *  - [ 'col', 'col2' ] which generates `col` = VALUES(`col`) and overwrites with the binding for the row.
     *  - [ 'col' => 'value' ] which generates `col` = ? and overwrites all rows with the specified value.
     *
     * @param  mixed[] $values
     * @param  mixed[] $update
     * @return bool
     */
    public static function insertOnDuplicateKey(array $values, array $update)
    {
        if (empty($values)) {
            return true;
        }

        [$query, $bindings] = self::compileInsert($values, false);

        $keys = [];
        if (isset($update[0])) {
            foreach ($update as $value) {
                $keys[] = sprintf('`%s` = VALUES(`%s`)', $value, $value);
            }
        } else {
            $self = new static;
            $grammar = $self->newQuery()->getQuery()->getGrammar();

            foreach ($update as $column => $value) {
                $parameter = $grammar->parameter($value);
                $keys[] = sprintf('`%s` = %s', $column, $parameter);

                if ($grammar->isExpression($value)) {
                    continue;
                }

                $bindings[] = $value;
            }
        }

        $query = trim($query) . ' ON DUPLICATE KEY UPDATE ' . implode(', ', $keys);

        return DB::insert($query, $bindings);
    }

    /**
     * Get all the model columns.
     *
     * @param  string[] $except     Exclude these columns.
     * @param  string[] $additional Include these that are not included in fillable.
     * @return array
     */
    public static function getColumnsExcept($except = [], $additional = [])
    {
        $class = new static;

        $columns = array_diff(
            array_merge(
                [$class->getKeyName(), $class->getCreatedAtColumn(), $class->getUpdatedAtColumn()],
                $class->getFillable(),
                $additional
            ),
            $except
        );

        return $columns;
    }

    /**
     * Unloads a given relation from the model.
     *
     * @param  string $relation
     * @return $this
     */
    public function unload($relation)
    {
        // Get current relations loaded
        $relations = $this->getRelations();

        // Remove from array
        unset($relations[$relation]);

        // Set it back on model
        $this->setRelations($relations);

        return $this;
    }

    /**
     * Checks if a query is joined to a given table name.
     *
     * @param  Builder $query
     * @param  string $table
     * @param  string $clause
     * @return bool
     */
    public static function isJoined($query, $table, $clause = null)
    {
        $joins = collect($query->getQuery()->joins);

        // If we want to check for a specific first clause
        if ($clause !== null) {
            return $joins->where('table', $table)
                ->pluck('clauses')
                ->flatten(1)
                ->contains('first', $clause);
        }

        return $joins->pluck('table')->contains($table);
    }

    /**
     * Compile an INSERT statement.
     *
     * @param  mixed[] $values
     * @param  bool    $ignore
     * @return array
     */
    private static function compileInsert(array $values, $ignore = false)
    {
        // Convert single dimensional array to multi dimensional array.
        // Allows us to use the function to insert a single record and also multiple records.
        if (! is_array(reset($values))) {
            $values = [$values];
        }

        $self = new static;
        $grammar = $self->newQuery()->getQuery()->getGrammar();

        $bindings = [];
        $now = $self->freshTimestampString();
        foreach ($values as $k => $v) {
            if ($self->timestamps) {
                $values[$k]['created_at'] = $now;
                $values[$k]['updated_at'] = $now;
            }

            foreach ($values[$k] as $value) {
                if ($grammar->isExpression($value)) {
                    continue;
                }

                $bindings[] = $value;
            }
        }

        $ignoreSql = $ignore ? 'IGNORE' : '';

        $table = $grammar->wrapTable($self->getTable());

        $columns = $grammar->columnize(array_keys(reset($values)));

        $placeholders = collect($values)->map(function ($record) use ($grammar) {
            return '(' . $grammar->parameterize($record) . ')';
        })->implode(', ');

        return [
            vsprintf('INSERT %s INTO %s (%s) VALUES %s', [
                $ignoreSql,
                $table,
                $columns,
                $placeholders
            ]),
            $bindings
        ];
    }

    /**
     * Get the namespace of the late static bound class (child class).
     *
     * @return string
     */
    public function namespace()
    {
        $fqcn = static::class;

        return substr($fqcn, 0, strrpos($fqcn, '\\'));
    }

    /**
     * Create a HTML element.
     *
     * @param  string  $name
     * @param  mixed[] $attributes
     * @param  string  $content
     * @return string
     */
    protected function element($name, array $attributes, $content)
    {
        $attributesToString = implode(' ', array_map(
            function ($v, $k) {
                return sprintf('%s="%s"', $k, $v);
            },
            $attributes,
            array_keys($attributes)
        ));

        return sprintf('<%s %s>%s</%s>', $name, $attributesToString, $content, $name);
    }
}
