<?php declare(strict_types=1);

namespace SupportPal\Core\Database\Traits;

use Illuminate\Database\Eloquent\Builder;

use function convert_ip_to_in_addr;
use function filter_var;
use function inet_ntop;
use function inet_pton;

use const FILTER_VALIDATE_IP;

trait HasIp
{
    /**
     * Set the User IP address to packed in_addr representation
     *
     * @param string|null $value
     */
    public function setIpAttribute(?string $value)
    {
        $this->attributes['ip'] = convert_ip_to_in_addr($value);
    }

    /**
     * Convert User IP address to human readable
     *
     * @param string|null $value
     * @return string|null
     */
    public function getIpAttribute(?string $value): ?string
    {
        // If for some reason we have an IP already, don't change it
        if (filter_var($value, FILTER_VALIDATE_IP)) {
            return $value;
        }

        if ($value !== null) {
            // If the value is not null and not an IP try to use inet_ntop
            return ($ip = @inet_ntop($value)) !== false ? $ip : null;
        }

        return null;
    }

    /**
     * Look for records by its IP address.
     *
     * @param  Builder $query
     * @param  string|null $value
     * @return Builder
     */
    public function scopeWhereIp(Builder $query, ?string $value): Builder
    {
        if (filter_var($value, FILTER_VALIDATE_IP)) {
            return $query->where('ip', @inet_pton($value));
        }

        return $query;
    }
}
