<?php declare(strict_types=1);

/**
 * File ResponseHelpers.php
 *
 * @package    SupportPal\Core\Http\Traits
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @since      File available since Release 2.0.0
 */
namespace SupportPal\Core\Http\Traits;

use Exception;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\ValidationException;

use function implode;
use function mb_strtolower;
use function strpos;

/**
 * Trait ResponseHelpers
 *
 * @package    SupportPal\Core\Http\Traits
 * @copyright  Copyright (c) 2015-2016 SupportPal (http://www.supportpal.com)
 * @license    http://www.supportpal.com/company/eula
 * @version    Release: @package_version@
 * @since      Class available since Release 2.0.0
 */
trait ResponseHelpers
{
    /**
     * Handle a generic controller error during an store routine.
     *
     * @param  string         $lang_key
     * @param  string         $route
     * @param  string[]|int[] $route_params
     * @return RedirectResponse
     */
    protected function genericStoreError($lang_key, $route, array $route_params = [])
    {
        $error_message = Lang::get('messages.error_created', ['item' => mb_strtolower($this->getLangVar($lang_key), 'UTF-8')]);

        return $this->genericError($error_message, $route, $route_params);
    }

    /**
     * Handle a generic controller error during an update routine.
     *
     * @param  string         $lang_key
     * @param  string         $route
     * @param  string[]|int[] $route_params
     * @return RedirectResponse
     */
    protected function genericUpdateError($lang_key, $route, array $route_params = [])
    {
        $error_message = Lang::get('messages.error_updated', ['item' => mb_strtolower($this->getLangVar($lang_key), 'UTF-8')]);

        return $this->genericError($error_message, $route, $route_params);
    }

    /**
     * Handle ModelNotFoundException.
     *
     * @param  string         $lang_key
     * @param  string         $route
     * @param  string[]|int[] $route_params
     * @return RedirectResponse
     */
    protected function notFoundError($lang_key, $route, array $route_params = [])
    {
        $error_message = Lang::get('messages.error_notfound', ['item' => mb_strtolower($this->getLangVar($lang_key), 'UTF-8')]);

        return $this->genericError($error_message, $route, $route_params);
    }

    /**
     * Flash a generic error message and redirect the user.
     *
     * @param  string         $error_message
     * @param  string         $route
     * @param  string[]|int[] $route_params
     * @return RedirectResponse|JsonResponse
     */
    protected function genericError($error_message, $route, array $route_params = [])
    {
        if (Request::ajax()) {
            return Response::json(['status' => 'error', 'message' => $error_message, 'data' => null], 200);
        }

        // Flash error for the page redirect.
        Session::flash('error', $error_message);

        if ($route instanceof RedirectResponse) {
            return $route->withInput();
        }

        return Redirect::route($route, $route_params)
            ->withInput();
    }

    /**
     * Handle validation errors.
     *
     * @param  Validator $validator
     * @param  string|RedirectResponse $route
     * @param  string[]|int[] $route_params
     * @return RedirectResponse|JsonResponse
     */
    protected function validationError($validator, $route, array $route_params = [])
    {
        if ($this->isJsValidation()) {
            return $this->jsValidationError($validator);
        }

        if (Request::ajax()) {
            return Response::json(
                [
                    'status'  => 'error',
                    'message' => implode(',', $validator->errors()->all()),
                    'data'    => null
                ],
                200,
                ['Content-Type' => 'application/json; charset=utf-8']
            );
        }

        if ($route instanceof RedirectResponse) {
            return $route->withErrors($validator)->withInput();
        }

        return Redirect::route($route, $route_params)
            ->withErrors($validator)
            ->withInput();
    }

    /**
     * Handle JS validation errors.
     *
     * @param  Validator $validator
     * @return JsonResponse
     */
    protected function jsValidationError($validator)
    {
        // Default set of errors
        $errors = implode(', ', $validator->errors()->all());

        // Try and get the error message for the field under validation
        $field_under_validation = Request::input('_jsvalidation', null);
        if (! empty($field_under_validation)) {
            $errors = $validator->errors()->first($field_under_validation);
        }

        return Response::json(
            (empty($errors) ? [] : [$errors]),
            200,
            ['Content-Type' => 'application/json; charset=utf-8']
        );
    }

    /**
     * Check whether the request is from the JS validation library.
     *
     * @return bool
     */
    private function isJsValidation()
    {
        return Request::filled('_jsvalidation');
    }

    /**
     * Get a language variable.
     *
     * @param  string $key
     * @return array|null|string
     */
    private function getLangVar($key)
    {
        $translated = Lang::get($key);

        return strpos($translated, '|') ? Lang::choice($key, 1) : $translated;
    }

    /**
     * If debug is enabled, returns the message of an exception, else gives null
     *
     * @param  Exception $e
     * @return null|string
     */
    protected function getExceptionMessage(Exception $e)
    {
        if ($e instanceof ValidationException) {
            return implode(', ', $e->validator->errors()->all());
        }

        if (Config::get('app.debug', false) === true) {
            return $e->getMessage();
        }

        return null;
    }
}
