<?php declare(strict_types=1);

namespace SupportPal\Marketplace\Http\Api\Addons;

use stdClass;
use SupportPal\Marketplace\Exceptions\AddonNotFound;
use SupportPal\Marketplace\Exceptions\ApiException;
use SupportPal\Marketplace\Http\Request;

use function sprintf;

class Addon
{
    protected Request $api;

    /**
     * @param Request $api
     */
    public function __construct(Request $api)
    {
        $this->api = $api;
    }

    /**
     * Get a single addon (cached for 3 hours).
     *
     * @param int $id
     * @return \SupportPal\Marketplace\Models\Addon\Addon
     * @throws AddonNotFound
     * @throws ApiException
     */
    public function get(int $id): \SupportPal\Marketplace\Models\Addon\Addon
    {
        try {
            $response = $this->api->get(sprintf('addons/%d', $id));
        } catch (ApiException $e) {
            if ($e->getCode() === 404) {
                throw new AddonNotFound($e->getMessage());
            }

            throw $e;
        }

        if (! isset($response->data)) {
            throw new AddonNotFound('Invalid response from marketplace.');
        }

        return $this->handleAddon($response->data);
    }

    /**
     * Get a single addon by name (cached for 3 hours).
     *
     * @throws AddonNotFound
     * @throws ApiException
     */
    public function find(string $category, string $identifier): \SupportPal\Marketplace\Models\Addon\Addon
    {
        try {
            $response = $this->api->get(sprintf('addons/%s/%s', $category, $identifier));
        } catch (ApiException $e) {
            if ($e->getCode() === 404) {
                throw new AddonNotFound($e->getMessage());
            }

            throw $e;
        }

        return $this->handleAddon($response->data);
    }

    /**
     * @param stdClass $data
     * @return \SupportPal\Marketplace\Models\Addon\Addon
     */
    private function handleAddon(stdClass $data): \SupportPal\Marketplace\Models\Addon\Addon
    {
        return new \SupportPal\Marketplace\Models\Addon\Addon($data);
    }
}
